/* #------------------------------------------------------------------------#
   |                                                                        |
   |   NWSWIS.C                                                             |
   |                                                                        |
   |   SWI entry points for NWClient.                                       |
   |                                                                        |
   |   Copyright 1999, Frank A. Vorstenbosch.                               |
   |                                                                        |
   #------------------------------------------------------------------------# */

/* $Id: nwswis.c,v 1.19 2001/12/22 12:19:51 frank Exp $ */

#define VERSION 0x00000

#include "nwclient.h"

#pragma On(Check_ltorg)

/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Attach to a file server.                                             |
   |                                                                        |
   +------------------------------------------------------------------------+ */

int *nwclient_attach(char *tree,int mode,register struct NWClient *nw)
{  register struct RiscIPX *ri=nw->ri;
   struct Socket *sk;
   struct Address addr={0,"\xff\xff\xff\xff\xff\xff",IPX_SKT_SAP};
   int i,r,tries;
   struct mbuf *mb;
   struct SAPEntry sappkt;
   char LoginDirPath[100];
   struct Server *fs;

   #if defined DEBUG
   Trace("\n attach (");
   Trace(tree);
   Trace(",");
   TraceByte(mode);
   Trace(") ");
   #endif
   
Trace(" nw->ri=");   TraceVal((int)(nw->ri));

   if((fs=malloc(sizeof(struct Server)))==NULL)
   {  r=IPX_NOT_ENOUGH_MEMORY;
      goto quick; }

   #ifdef DEBUG
   for(i=0;i<sizeof(struct Server);i+=4)
      *(int *)(((char *)fs)+i)='vreS';
   #endif

   NewList(&fs->Files);
   fs->Flags=0;
   fs->nw=nw;
   fs->BlockSize=512;
   fs->Timeout=750/40;   /* initially 750ms */

   if(mode==ATTACH_NODE)
   {  
      fs->Address.Network=0;
      memcpy(fs->Address.Node,"\0\0\0\0\0\1",6);
      fs->Address.Port=IPX_SKT_NCP;
      if((r=AtoIPX(&fs->Address,tree))!=0 || fs->Address.Network==0)
      {  r=IPX_INVALID_ADDRESS;
         goto freefs; }
   }
   else if(mode==ATTACH_TCPIP)
   {
      fs->Address.Network=0; /* will store IP address */
      fs->Address.Port=524;
      
      if((fs->Address=inet_aton(tree))==0)
      {  r=IPX_INVALID_ADDRESS;
         goto freefs; }

      fs->Flags|=SERVER_TCP_IP;
      fs->MsgSocket=fs->WdogSocket=0;

      nw->Flags|=NWC_DONT_STREAM;
   }
   else
   {
      if((sk=ipx_socket(ri))==NULL)
      {  r=IPX_NO_SOCKET;
         goto freefs; }

      if((r=ipx_bind(sk,0,0,ri))!=0 ||
         (r=ipx_connect(sk,IPX_PT_NCP,&addr,ri))!=0)
      {  ipx_close(sk,ri);
         #if defined DEBUG && (DEBUG&4)
         ShowError("attach_1",r);
         #endif
         goto freefs; }

      fs->Address.Port=0;

      for(tries=0;tries<100 && fs->Address.Port==0;tries++)
      {
         #if defined DEBUG && DEBUG&4
         Trace("\n try ");
         #endif

         if((tries&1)==0)
         {
            char msg[4*4]={ SAPTYPE_QUERY_NEAREST>>8,SAPTYPE_QUERY_NEAREST&255,NCP_TYPE_NDS_SERVER>>8,NCP_TYPE_NDS_SERVER&255,
                            SAPTYPE_QUERY_NEAREST>>8,SAPTYPE_QUERY_NEAREST&255,NCP_TYPE_386_SERVER>>8,NCP_TYPE_386_SERVER&255,
                            SAPTYPE_QUERY_NEAREST>>8,SAPTYPE_QUERY_NEAREST&255,NCP_TYPE_NDS_SERVER>>8,NCP_TYPE_NDS_SERVER&255,
                            SAPTYPE_QUERY_NEAREST>>8,SAPTYPE_QUERY_NEAREST&255,NCP_TYPE_FILESERVER>>8,NCP_TYPE_FILESERVER&255 };

            if((r=ipx_send_unsafe(sk,msg+((tries&24)>>1),4,ri))!=0 && r!=IPX_NO_ROUTE_TO_NETWORK)
            {  ipx_close(sk,ri);
               goto freefs; }
         }

         sleep10ms();

         if((mb=ipx_recv_mbuf(sk,ri))!=NULL)
         {
            #if defined DEBUG && DEBUG&2
            Trace("\n mbuf ");
            #endif

            while((i=UnpackN(mb->next,&sappkt,"WW48bA",NULL,mb->pkthdr.len))>0)
            {  mb->pkthdr.len+=i;

               if(sappkt.PacketResponseType!=SAPTYPE_RESPONSE &&
                  sappkt.PacketResponseType!=SAPTYPE_RESPONSE_NEAREST &&
                  sappkt.ServerType!=NCP_TYPE_NDS_SERVER)
                  continue;

               sappkt.Name[32]=0;
               for(i=31;i>=0;i--)
               {  if(sappkt.Name[i]=='_')
                     sappkt.Name[i]=0;
                  else
                     break; }

               #if defined DEBUG && DEBUG&2
               Trace("\n SAP ");
               Trace(sappkt.Name);
               #endif

               if((!tree && sappkt.PacketResponseType!=SAPTYPE_RESPONSE_NEAREST) || !strcmp(tree,sappkt.Name))
               {  
                  Trace(" got server address");

                  memcpy(&fs->Address,&sappkt.Server,sizeof(struct Address));
                  fs->Address.Port=IPX_SKT_NCP;
                  break;
               }
            } /* while(sap entries) */

            ri->mbctl.freem(&ri->mbctl,mb);
         } /* if(got a mb) */
      } /* for(tries) */

      ipx_close(sk,ri);

      if(fs->Address.Port==0)
      {  r=NCP_TIMEOUT_FAILURE;
         goto freefs; }
   }

   r=IPX_NO_SOCKET;
   if(fs->Flags&SERVER_TCP_IP)
   {  int sk=Socket_Create(2,1,0);
      if(sk!=-1)
      {  fs->NcpSocket=(struct Socket *)sk;

         UNIM: connect to server

         r=0;
      }
   }
   else
      for(tries=0;tries<100 && r;tries+=10)
      {
         #ifdef DEBUG
         Trace("\nTry getting sockets");
         #endif

         if(r=IPX_NO_SOCKET,(fs->NcpSocket=ipx_socket(ri))!=NULL)
         {  fs->Address.Port=IPX_SKT_NCP;
            if((r=ipx_bind(fs->NcpSocket,IPX_PT_NCP,0,ri))!=0 ||
               (r=ipx_connect(fs->NcpSocket,IPX_PT_NCP,&fs->Address,ri))!=0)
               {  ipx_close(fs->NcpSocket,ri);
                  fs->NcpSocket=NULL; }
         }

         i=ipx_getport(fs->NcpSocket);
         #ifdef DEBUG
         TraceVal(i);
         TraceVal(i+1);
         TraceVal(i+2);
         #endif

         if(r)
         {  tries++;
            continue; }

         if(!r)
         {
            if(r=IPX_NO_SOCKET,(fs->WdogSocket=ipx_socket(ri))!=NULL)
            {  fs->Address.Port=0;
               if((r=ipx_bind(fs->WdogSocket,0/*IPX_PT_PEP*/,i+1,ri))!=0 ||
                  (r=ipx_connect(fs->WdogSocket,IPX_PT_PEP,&fs->Address,ri))!=0 ||
                  (r=ipx_recv_handler(fs->WdogSocket,&wdog_handler,fs,ri))!=0)
               {  ipx_close(fs->WdogSocket,ri);
                  fs->WdogSocket=NULL; }
            }
         }

         if(!r)
         {
            if(r=IPX_NO_SOCKET,(fs->MsgSocket=ipx_socket(ri))!=NULL)
            {  fs->Address.Port=0;
               if((r=ipx_bind(fs->MsgSocket,0/*IPX_PT_PEP*/,i+2,ri))!=0 ||
                  (r=ipx_connect(fs->MsgSocket,IPX_PT_PEP,&fs->Address,ri))!=0 ||
                  (r=ipx_recv_handler(fs->MsgSocket,&wdog_handler,fs,ri))!=0)
               {  ipx_close(fs->MsgSocket,ri);
                  fs->MsgSocket=NULL; }
            }
         }

         i+=16;
         if(i>LASTPORT)
            i=FIRSTPORT;
      }

   if(r)
      goto freefs;
   
   /* fs->Address.Port=IPX_SKT_NCP; */

   fs->request.header.Type=NCP_ALLOC_SLOT_REQUEST;
   fs->request.header.Sequence=fs->request.header.Task=fs->request.header.Function=0;
   fs->ConnLow=fs->ConnHigh=fs->request.header.ConnLow=fs->request.header.ConnHigh=0xff;

   #if defined DEBUG && DEBUG&2
   Trace("\n try allocating a connection slot");
   #endif

   if((r=ncp_request("",NULL,NULL,fs))<0)
      goto abort;   /* could not get connection for some reason */

   fs->request.header.Type=NCP_REQUEST;
   fs->ConnLow=fs->request.header.ConnLow=nw->reply.header.ConnLow;
   fs->ConnHigh=fs->request.header.ConnHigh=nw->reply.header.ConnHigh;

   fs->request.header.Function=0x17;
   fs->request.payload.fn_17.Length=1;
   fs->request.payload.fn_17.SubFunction=0x11;     /* get file server info */
   ncp_request(REQ_17,"48b",NULL,fs);
   Trace("\n server=");
   Trace(nw->reply.payload.fn_17_11.Name);
   fs->ServerName=strdup(nw->reply.payload.fn_17_11.Name);
   fs->UserName=NULL;
   
   AddTail(&nw->Servers,fs);
   nw->NumServers++;

   Trace(" MountList=");
   TraceVal((int)(&nw->Mounts));

   if(IsEmptyList(&nw->Mounts))
   {  fs->request.header.Function=0x16;   /* get path to directory #1 */
      if((r=ncp_request("zk\002k\001k\001","c",LoginDirPath,fs))>=0)
      {  
         strcpy(nw->SmallBuffer,fs->ServerName);
         strcat(nw->SmallBuffer,"/");
         strcat(nw->SmallBuffer,LoginDirPath);

         #ifdef DEBUG
         Trace("\n Login directory=");
         Trace(nw->SmallBuffer);
         #endif
         nwclient_mount("Login",nw->SmallBuffer,MOUNT_LONGNAMES|MOUNT_AUTOMOUNT,nw);
   }  }

   #if defined DEBUG && DEBUG&2
   Trace(" fs=");
   TraceVal((int)fs);
   #endif

   ServiceCall(SERVICE_ATTACH,NWCLIENT_SERVICE_CALL,fs);

   nw->ReturnStruct[0]=0;
   nw->ReturnStruct[1]=(int)fs;
   return nw->ReturnStruct;

abort:
   if(fs->Flags&SERVER_TCP_IP)
      Socket_Close((int)fs->NcpSocket);
   else
   {  ipx_close(fs->MsgSocket,ri);
      ipx_close(fs->WdogSocket,ri);
      ipx_close(fs->NcpSocket,ri); }
freefs:
   free(fs);
quick:

   nw->ReturnStruct[0]=(int)LookupError(r);
   nw->ReturnStruct[1]=0;
   return nw->ReturnStruct;
}

/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Detach from a file server.                                           |
   |                                                                        |
   +------------------------------------------------------------------------+ */

int nwclient_detach(struct Server *fs,register struct NWClient *nw)
{  struct Mount *mt,*next;

   #if defined DEBUG
   Trace("\n detach ");
   #endif

   forList2(&nw->Mounts,mt,next)
   {  if(mt->Server==fs)
      {  mt->MountCount=1;
         nwclient_dismount(mt,nw); }
   }

   if(fs->Flags&SERVER_LOGGED_IN)
      nwclient_logout(fs,nw);

   ServiceCall(SERVICE_DETACH,NWCLIENT_SERVICE_CALL,fs);

   if(fs->NcpSocket)
   {  
      fs->request.header.Type=NCP_DEALLOC_SLOT_REQUEST;
      fs->request.header.Function=0;
      ncp_request("",NULL,NULL,fs);

      if(fs->Flags&SERVER_TCP_IP)
         Socket_Close((int)fs->NcpSocket);
      else
         ipx_close(fs->NcpSocket,nw->ri);
   }

   if(fs->WdogSocket)
      ipx_close(fs->WdogSocket,nw->ri);

   if(fs->MsgSocket)
      ipx_close(fs->MsgSocket,nw->ri);

   nw->NumServers--;
   RemoveNode(fs);

   free(fs->ServerName);
   free(fs);
   return 0;
}


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Log in to a (attached) file server.                                  |
   |                                                                        |
   +------------------------------------------------------------------------+ */

int nwclient_login(const char *name,char *password,register struct NWClient *nw)
{  int r,loginerr;
   char *p;
   struct Server *fs;
   struct user_lookup *user;

   #if defined DEBUG
   Trace("\n nwclient_login \n");
   #endif

   if((fs=lookup_server(name,&name,nw))==NULL)     /* not attached to that server */
      return IPX_NOT_ATTACHED;

   /* ----- uppercase password ------------------------------------------------- */

   if(password)
   {  for(p=password;*p;p++)
         if(*p>='a' && *p<='z')
            *p-='a'-'A';
   }
   else
      password="";

   /* -------------------------------------------------------------------------- */

   if(fs->UserName)
      free(fs->UserName);

	if((loginerr=ncp_login_user(fs,name,password))<0 && loginerr!=NCP_PASSWORD_EXPIRED)
      return loginerr;

   #if defined DEBUG && DEBUG&2
   Trace(" loginerr=");
   TraceVal(loginerr);
   #endif

   /* -------------------------------------------------------------------------- */

   #if 0
   fs->request.header.Function=0x17;
   fs->request.payload.fn_17.Length=1;
   fs->request.payload.fn_17.SubFunction=0x11;     /* get file server info */
   ncp_request(REQ_17,"48b",NULL,fs);
   Trace("\n server=");
   Trace(nw->reply.payload.fn_17_11.Name);
   #endif

   Trace("\n here fs=");
   TraceVal((int)fs);
   Trace(" nw=");
   TraceVal((int)nw);

   nw->TimeOffset=ReadCurrentTimeZone()/100;  /* just to keep up to date if user changes timezone */
   nwclient_systime(fs,nw);

   /* ----- request larger packets --------------------------------------------- */

   if((fs->Flags&SERVER_TCP_IP) || ipx_get_mtu(nw->ri)>1100)
   {
      fs->request.header.Function=0x21;
      fs->request.payload.fn_21.Length=1500;
      if(ncp_request("W","W",NULL,fs)>=0)
      {  if(nw->reply.payload.fn_21.Length>=1024)
         {  
            #if defined DEBUG && DEBUG&2
            Trace("Block size now 1024");
            #endif
            fs->BlockSize=1024;
   }  }  }

   /* -------------------------------------------------------------------------- */

   #if defined DEBUG && 0
   {  int seq=-1,n=0,i;
      struct scan_property *p;
      char *props[100];

      do
      {  if(nwclient_scanprop(NCP_TYPE_USER,name,seq,&p,fs))
            break;

         TraceChar('\n');
         Trace("Property ");
         Trace(p->Name);
         Trace("  Flags,Security,Value=");
         TraceByte(p->Flags);
         TraceByte(p->Security);
         TraceByte(p->Value);
         props[n]=strdup(p->Name);
         n++;

         seq=p->Sequence;
      } while(seq!=-1);

      for(i=0;i<n;i++)
      {  int segment;
         struct read_property *q;

         Trace("\n Reading property ");
         Trace(props[i]);

         for(segment=1;;segment++)
         {
            if(nwclient_readprop(NCP_TYPE_USER,name,props[i],segment,&q,fs))
               break;

            Trace(" Value=");
            Trace(q->Value);
            TraceChar('\n');

            if(!q->More)
               break;
         }
         free(props[i]);
      }
   }
   #endif

   if((r=ncp_connection_info(&user,fs->ConnLow|(fs->ConnHigh<<8),fs,nw))==0)
      fs->UserName=strdup(user->Name);

   fs->Flags|=SERVER_LOGGED_IN;
   ServiceCall(SERVICE_LOGIN,NWCLIENT_SERVICE_CALL,fs);

   Trace("\n login finished fs=");
   TraceVal((int)fs);
   Trace(" nw=");
   TraceVal((int)nw);
      
   return loginerr;
}


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Log out from a file server.                                          |
   |                                                                        |
   +------------------------------------------------------------------------+ */

int nwclient_logout(struct Server *fs,register struct NWClient *nw)
{  struct Mount *mt;
   struct File *file;
   void *next;

   #if defined DEBUG
   Trace("\n nwclient_logout \n");
   #endif

   forList2(&fs->Files,file,(struct File *)next)
      nwclient_close(file,nw);

   forList2(&nw->Mounts,mt,(struct Mount *)next)
   {  if(mt->Server==fs)
      {  mt->MountCount=1;
         if((mt->Flags&MOUNT_AUTOMOUNT)==0)
            nwclient_dismount(mt,nw); 
      }
   }

   ServiceCall(SERVICE_LOGOUT,NWCLIENT_SERVICE_CALL,fs);

   fs->Flags&=~SERVER_LOGGED_IN;
   if(fs->UserName)
   {  free(fs->UserName);
      fs->UserName=NULL; }
	
   fs->request.header.Function=0x19;
   return ncp_request("",NULL,NULL,fs);
}


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Mount a file server path as a local disc.                            |
   |                                                                        |
   +------------------------------------------------------------------------+ */

int *nwclient_mount(const char *localName,const char *remoteName,int flags,register struct NWClient *nw)
{  struct Mount *mt;
   struct Server *fs;
   int i,r,mtflag;
   char *p;

   #if defined DEBUG
   Trace("\n nwclient_mount(");
   if(remoteName)
      Trace(remoteName);
   Trace(") ");
   #endif

   #if defined DEBUG && DEBUG&2
   Trace(" nw=");
   TraceVal((int)nw);
   #endif

   if((fs=lookup_server(remoteName,&remoteName,nw))==NULL)     /* not attached to that server */
   {  nw->ReturnStruct[0]=(int)LookupError(IPX_NOT_ATTACHED);
      return nw->ReturnStruct; }

   #if defined DEBUG && DEBUG&2
   Trace(" in mount fs=");
   TraceVal((int)fs);
   Trace("=");
   Trace(fs->ServerName);
   Trace(" fs->NcpSocket=");
   TraceVal((int)(fs->NcpSocket));
   #endif

   mtflag=flags&(MOUNT_AUTOMOUNT|MOUNT_USERFLAGS);

   /* ----- try an existing mount ---------------------------------------------- */

   Trace(" try existing mount");

   Trace(" MountList=");
   TraceVal((int)(&nw->Mounts));

   forList(&nw->Mounts,mt)
   {  
      TraceVal((int)mt);

      if(!strcmp(mt->LocalName,localName))   /* if same name name */
      {  if((remoteName && strcmp(mt->RemoteName,remoteName)) || mt->Server!=fs)
         {  nw->ReturnStruct[0]=(int)LookupError(NCP_PARAMETERS_INVALID);   /* error if different remote server or name */
            return nw->ReturnStruct; }
         else
         {  mt->MountCount++;                   /* else increment mount count */
            mt->Flags=(mt->Flags&~MOUNT_USERFLAGS)|(flags&MOUNT_USERFLAGS);
            nw->ReturnStruct[0]=0;
            nw->ReturnStruct[1]=(int)mt;
            return nw->ReturnStruct; }
      }
   }

   if(!remoteName)
   {  nw->ReturnStruct[0]=(int)LookupError(NCP_PARAMETERS_INVALID);
      return nw->ReturnStruct; }

   /* ----- no, must make a new mount ------------------------------------------ */

   if((mt=malloc(sizeof(struct Mount)))==NULL)
   {  nw->ReturnStruct[0]=(int)LookupError(IPX_NOT_ENOUGH_MEMORY);
      return nw->ReturnStruct; }

   #if defined DEBUG && DEBUG&2
   Trace(" new mt=");
   TraceVal((int)mt);
   #endif
   mt->Flags=mtflag;
   mt->MountCount=1;
   mt->ScanDirName=NULL;
   NewList(&mt->ScanDirCache);
   mt->Server=fs;

   mt->LocalName=strdup(localName);
   mt->RemoteName=strdup(remoteName);

   NewList(&mt->Files);

   /* ----- generate directory handle ------------------------------------------ */

   for(i=((flags&MOUNT_LONGNAMES)?0:4);i>=0;i-=4)
   {  fs->request.header.Function=0x57;
      fs->request.payload.fn_57.SubFunction=0x16;     /* generate directory handle */
      fs->request.payload.fn_57_16.NameSpace=i;       /* namespace (0 for DOS or 4 for OS/2 */
      fs->request.payload.fn_57_16.Volume=-1;         /* faked volume number */
      fs->request.payload.fn_57_16.Directory=0;       /* faked dir_base */
      fs->request.payload.fn_57_16.HandleFlag=-1;     /* Don't have a dir_base */
      fs->request.payload.fn_57_16.PathBytes=make_path(&fs->request.payload.fn_57_16.PathComponents,mt->RemoteName);
      if((r=ncp_request(REQ_57"b3z"REQ_PATH,"llb",NULL,fs))==0)
         break;
      p=mt->RemoteName;
      while(*p)
      {  if(*p>='a' && *p<='z')
            *p-=32;
         p++; }
   }

   mt->DosDirectory=nw->reply.payload.fn_57_16.DosDirectory;
   mt->Directory=nw->reply.payload.fn_57_16.Directory;
   mt->Volume=nw->reply.payload.fn_57_16.Volume;

   if(r<0)
   {
      #if defined DEBUG && (DEBUG&4)
      ShowError("gendirhandle",r);
      #endif
      free(mt);
      nw->ReturnStruct[0]=(int)LookupError(r);
      return nw->ReturnStruct;
   }

   /* ----- see if we have long file names ------------------------------------- */

   if(i==4)
   {  mt->Flags|=MOUNT_LONGNAMES;
      #if defined DEBUG && (DEBUG&2)
      Trace("\n long file names on mount point ");
      Trace(localName);
      #endif
   }

   /* -------------------------------------------------------------------------- */

   AddTail(&nw->Mounts,mt);

   #if defined DEBUG && DEBUG&4
   Trace(" good mount");
   #endif

   ServiceCall(SERVICE_MOUNT,NWCLIENT_SERVICE_CALL,mt);

   nw->ReturnStruct[0]=0;
   nw->ReturnStruct[1]=(int)mt;
   return nw->ReturnStruct;
}


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Dismount a local disc name from the file server path.                |
   |                                                                        |
   +------------------------------------------------------------------------+ */

int nwclient_dismount(struct Mount *mt,register struct NWClient *nw)
{  struct File *file,*next;
   
   #if defined DEBUG && (DEBUG&4)
   Trace("\n nwclient_dismount");
   TraceVal((int)mt);
   #endif

   if(!mt || --mt->MountCount!=0)
      return 0;

   if(mt==nw->CurrentMount)
      nw->CurrentMount=NULL;

   FlushDirCache(mt,nw);

   forList2(&mt->Files,file,next)
      nwclient_close(file,nw);

   ServiceCall(SERVICE_DISMOUNT,NWCLIENT_SERVICE_CALL,mt);

   RemoveNode(mt);
   free(mt);

   return 0;
}


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Change a user's password.                                            |
   |                                                                        |
   +------------------------------------------------------------------------+ */

int nwclient_setpass(const char *username,char *oldpass,char *newpass,register struct NWClient *nw)
{  int r,uid;
	char *p,ncp_key[8],namebuf[49];
   struct user_lookup *user;
   unsigned char oldpwd[16];       /* old passwd as stored by server */
   struct ncp_bindery_object other;   
   struct Server *fs;

   #if defined DEBUG && (DEBUG&4)
   Trace("nwclient_setpass(,\"");
   Trace(username);
   Trace("\",\"");
   Trace(oldpass);
   Trace("\",\"");
   Trace(newpass);
   Trace("\") ");
   #endif

   if((fs=lookup_server(username,&username,nw))==NULL)     /* not attached to that server */
      return IPX_NOT_ATTACHED;

   if(username && *username)
   {
      if((r=ncp_get_bindery_object_id(fs,NCP_TYPE_USER,username,&other))<0)
         return r;

      uid=htonl(other.object_id);
      username=other.object_name;
   }
   else
   {
      if((r=ncp_connection_info(&user,fs->ConnLow|(fs->ConnHigh<<8),fs,nw))<0)
         return r;

      uid=user->UserID;
      strcpy(namebuf,user->Name);
      username=namebuf;
   }

   /* ----- uppercase passwords ------------------------------------------------ */

   if(oldpass)
   {  for(p=oldpass;*p;p++)
         if(*p>='a' && *p<='z')
            *p-='a'-'A';
   }
   else
      oldpass="";

   if(newpass)
   {  for(p=newpass;*p;p++)
         if(*p>='a' && *p<='z')
            *p-='a'-'A';
   }
   else
      newpass="";

   /* -------------------------------------------------------------------------- */
   
   if((r=ncp_get_encryption_key(fs,ncp_key))==NCP_NCP_NOT_SUPPORTED)
   {
      /* change password in clear text */

      fs->request.header.Function=0x17;
      fs->request.payload.fn_17.Length=1+2+1+strlen(username)+1+strlen(oldpass)+1+strlen(newpass);
      fs->request.payload.fn_17.SubFunction=0x40;
      fs->request.payload.fn_17_40.Type=NCP_TYPE_USER;
      fs->request.payload.fn_17_40.Name=username;
      fs->request.payload.fn_17_40.OldPassword=oldpass;
      fs->request.payload.fn_17_40.NewPassword=newpass;

      r=ncp_request(REQ_17"Wccc",NULL,NULL,fs);
   }
   else
   {  shuffle((unsigned const char *)&uid,(unsigned const char *)oldpass,strlen(oldpass),oldpwd);
      shuffle((unsigned const char *)&uid,(unsigned const char *)newpass,strlen(newpass),fs->request.payload.fn_17_4b.NewPass);

      nw_encrypt((unsigned const char *)ncp_key,oldpwd,fs->request.payload.fn_17_4b.Key);
      
      newpassencrypt((char *)oldpwd,(char *)fs->request.payload.fn_17_4b.NewPass,(char *)fs->request.payload.fn_17_4b.NewPass);
      newpassencrypt((char *)oldpwd+8,(char *)fs->request.payload.fn_17_4b.NewPass+8,(char *)fs->request.payload.fn_17_4b.NewPass+8);

      if((fs->request.payload.fn_17_4b.PassLen=strlen(newpass))>63)
         fs->request.payload.fn_17_4b.PassLen=63;
      fs->request.payload.fn_17_4b.PassLen=((fs->request.payload.fn_17_4b.PassLen^oldpwd[0]^oldpwd[1])&0x7f)|0x40;

      fs->request.header.Function=0x17;
      fs->request.payload.fn_17.Length=1+8+2+1+16+1+strlen(username);
      fs->request.payload.fn_17.SubFunction=0x4b;
      fs->request.payload.fn_17_4b.Type=NCP_TYPE_USER;
      fs->request.payload.fn_17_4b.Name=username;

      r=ncp_request(REQ_17"8bWc17b",NULL,NULL,fs);
   }

   return r;
}


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   List attached servers.                                               |
   |                                                                        |
   +------------------------------------------------------------------------+ */

int *nwclient_serverlist(struct Server *fs,register struct NWClient *nw)
{  struct Server *tmp;

   nw->ReturnStruct[0]=0;

   if(!fs)
   {  if(!FirstNode(&nw->Servers,fs))
         return nw->ReturnStruct;
   }
   else
   {  int found=0;
      forList(&nw->Servers,tmp)
      {  if(tmp==fs)
         {  found=1;
            NextNode(fs);
            break; }
      }
      if(!found)
      {  nw->ReturnStruct[0]=-1;
         nw->ReturnStruct[1]=(int)LookupError(NCP_DIRHANDLE_INVALID);
         return nw->ReturnStruct; }
   }

   if(IsTailNode(fs))
      return nw->ReturnStruct;
   
   nw->ReturnStruct[0]=(int)fs;
   nw->ReturnStruct[1]=(int)(fs->ServerName);
   nw->ReturnStruct[2]=(int)(fs->UserName);
   nw->ReturnStruct[3]=fs->Flags;

   return nw->ReturnStruct;
}


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   List mount points.                                                   |
   |                                                                        |
   +------------------------------------------------------------------------+ */

int *nwclient_mountlist(struct Mount *mt,register struct NWClient *nw)
{  struct Mount *tmp;

   nw->ReturnStruct[0]=0;

   if(!mt)
   {  if(!FirstNode(&nw->Mounts,mt))
         return nw->ReturnStruct;
   }
   else
   {
      int found=0;
      forList(&nw->Mounts,tmp)
      {  if(tmp==mt)
         {  found=1;
            NextNode(mt);
            break; }
      }
      if(!found)
      {  nw->ReturnStruct[0]=-1;
         nw->ReturnStruct[1]=(int)LookupError(NCP_DIRHANDLE_INVALID);
         return nw->ReturnStruct; }
   }

   if(IsTailNode(mt))
      return nw->ReturnStruct;
   
   nw->ReturnStruct[0]=(int)mt;
   nw->ReturnStruct[1]=(int)(mt->LocalName);
   nw->ReturnStruct[2]=(int)(mt->RemoteName);
   nw->ReturnStruct[3]=mt->Flags;
   if((nw->ReturnStruct[4]=mt->MountCount)==0)
      return NULL;
   nw->ReturnStruct[5]=(int)(mt->Server);

   return nw->ReturnStruct;
}


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Low level NCP request.                                               |
   |                                                                        |
   +------------------------------------------------------------------------+ */

int nwclient_ncpreq(struct Server *fs,int function,void *request,char *reqfmt,void *reply,char *rplfmt,char *strings,register struct NWClient *nw)
{  int r;

   #if defined DEBUG
   Trace("\n ncpreq(");
   TraceByte(function);
   TraceVal((int)request);
   TraceVal(*(int *)((char *)request+0));
   TraceVal(*(int *)((char *)request+4));
   TraceVal(*(int *)((char *)request+8));
   TraceVal(*(int *)((char *)request+12));
   Trace(") ");
   #endif

   fs->request.header.Function=function;
   memcpy(&fs->request.payload.Data[0],request,256);

   if((r=_ncp_request(reqfmt,rplfmt,strings,fs))==0)
      memcpy(reply,&nw->reply.payload.Data[0],nw->ReplyLength);

   return r;
}


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Set RiscOS time from file server time.                               |
   |                                                                        |
   +------------------------------------------------------------------------+ */

extern int day_n[];

int nwclient_systime(struct Server *fs,register struct NWClient *nw)
{  int r,month,day,year,hour,minute,second;
   int block[2];

   /* ----- get file server time ----------------------------------------------- */

   fs->request.header.Function=0x14;
   if((r=ncp_request("","7b",NULL,fs))<0)
      return r;

   year=nw->reply.payload.Data[0]-80;
   month=nw->reply.payload.Data[1]-1;
   day=nw->reply.payload.Data[2]-1;
   hour=nw->reply.payload.Data[3];
   minute=nw->reply.payload.Data[4];
   second=nw->reply.payload.Data[5];

   #if defined DEBUG && (DEBUG&4)
   Trace("\n year  =");   TraceVal(year  );
   Trace("\n month =");   TraceVal(month );
   Trace("\n day   =");   TraceVal(day   );
   Trace("\n hour  =");   TraceVal(hour  );
   Trace("\n minute=");   TraceVal(minute);
   Trace("\n second=");   TraceVal(second);
   #endif

   /* ----- convert to UNIX time ----------------------------------------------- */

	second=second+60*minute+3600*hour+86400*
	   (day+day_n[month]+(year/4)+year*365-((year&3) == 0 && month<2?1:0)+3653);

   #if defined DEBUG && (DEBUG&4)
   Trace("\n UNIX time  =");   TraceVal(second);
   #endif

   /* ----- convert to RiscOS time --------------------------------------------- */

   #define DELTASEC (60U*60U*24U*(365U*70U+68U/4U))
   mul100(block,second+DELTASEC);

   #if defined DEBUG && (DEBUG&4)
   Trace("\n block[0]=");   TraceVal(block[0]);
   Trace("\n block[1]=");   TraceVal(block[1]);
   #endif

   SetROTime(block);
   
   return 0;
}


/* ----- EOF NWSWIS.C ----- */

