/* #------------------------------------------------------------------------#
   |                                                                        |
   |   NWCLIENT.H                                                           |
   |                                                                        |
   |   NWClient data structures.                                            |
   |                                                                        |
   |   Copyright 1998, Frank A. Vorstenbosch.                               |
   |                                                                        |
   #------------------------------------------------------------------------#

   File created 16-dec-98   */

#include "pragmas.h"
#include "../libmod.h"
#include "riscipx.h"
#include "lists.h"
#include "errors.h"
#include "services.h"
#include "sap.h"
#include "debug.h"


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   NCP request and reply.                                               |
   |                                                                        |
   +------------------------------------------------------------------------+ */

struct ncp_request_header
{  short Type;
      #define NCP_ALLOC_SLOT_REQUEST   0x1111
      #define NCP_REQUEST              0x2222
      #define NCP_DEALLOC_SLOT_REQUEST 0x5555
      #define NCP_REPLY                0x3333
      #define NCP_PACKET_BURST         0x7777
      #define NCP_POSITIVE_ACK         0x9999
   char Sequence,
        ConnLow,
        Task,
        ConnHigh,
        Function;
};

struct ncp_reply_header
{  short Type;
   char Sequence,
        ConnLow,
        Task,
        ConnHigh,
        CompletionCode,
        ConnectionState;
   #define CONSTATE_DISCONNECTED    0x10
   #define CONSTATE_MESSAGE         0x40
};

struct stdrequest
{  struct ncp_request_header header;
   union
   {  char Data[1536];
      struct
      {  char Volume;
      } fn_12;
      struct
      {  short Length;
         char SubFunction;
      } fn_15;
      struct
      {  short Length;
         char SubFunction;
      } fn_16;
      struct
      {  short Length;
         char SubFunction,
              Volume;
      } fn_16_06;
      struct
      {  short Length;
         char SubFunction,
              Volume;
      } fn_16_2c;
      struct
      {  short Length;
         char SubFunction;
         int Volume;
      } fn_16_33;
      struct
      {  short Length;
         char SubFunction;
         int FirstVolume;
      } fn_16_34;
      struct
      {  short Length;
         char SubFunction;
      } fn_17;
      struct
      {  short Length;
         char SubFunction;
         short Type;
         const char *Name;
         const char *Password;
      } fn_17_14;
      struct
      {  short Length;
         char SubFunction,
              Connection;
      } fn_17_16;
      struct
      {  short Length;
         char SubFunction;
         char Password[8];
         short Type;
         const char *Name;
      } fn_17_18;
      struct
      {  short Length;
         char SubFunction;
         int Connection;
      } fn_17_1c;
      struct
      {  short Length;
         char SubFunction;
         short Type;
         const char *Name;
      } fn_17_35;
      struct
      {  short Length;
         char SubFunction;
         int Sequence;
         short Type;
         const char *Name;
      } fn_17_37;
      struct
      {  short Length;
         char SubFunction;
         short Type;
         const char *Name;
         int Sequence;
         const char *PropName;
      } fn_17_3c;
      struct
      {  short Length;
         char SubFunction;
         short Type;
         const char *ObjectName;
         int Segment;
         const char *PropName;
      } fn_17_3d;
      struct
      {  short Length;
         char SubFunction;
         short Type;
         const char *Name,
                    *OldPassword,
                    *NewPassword;
      } fn_17_40;
      struct
      {  short Length;
         char SubFunction;
         unsigned char Key[8];
         short Type;
         const char *Name;
         unsigned char PassLen,
                       NewPass[16];
      } fn_17_4b;
      struct
      {  short Length;
         char SubFunction;
         int QueueID;
         struct queue_entry
         {  short InUse;
            int prevEntry,
                nextEntry,
                ClientStation,
                ClientTask,
                ClientObjectID,
                TargetServerID;
            char TargetExecTime[6];
            char JobEntryTime[6];
            int JobNumber;
            short JobType,
                  JobPosition,
                  JobControlFlags;
            char FileNameLen;
            char JobFileName[13];
            int JobFileHandle,
                ServerStation,
                ServerTaskNumber,
                ServerObjectID;
         } Entry;
         char JobTextDescription[50];
         struct print_job_record
         {  char Version,
                 TabSize;
            short Copies,
                  CtrlFlags,
                  Lines,
                  Columns;
            char FormName[16],
                 BannerName[13],
                 FnameBanner[13],
                 FnameHeader[14],
                 Path[80];
         } ClientRecordArea;
      } fn_17_79;
      struct
      {  short Length;
         char SubFunction;
         int Queue,
             Job;
      } fn_17_7f;
      struct
      {  short Length;
      } fn_21;
      struct
      {  char Handle[6];
      } fn_42;
      struct
      {  char Handle[6];
      } fn_47;
      struct
      {  char Handle[6];
         int Offset;
         short Length;
      } fn_48;
      struct
      {  char Handle[6];
         int Offset;
         short Length;
         char Data[1];
      } fn_49;
      struct
      {  char SubFunction;
         int EAHandle;
      } fn_56_01;
      struct
      {  char SubFunction;
         short Flags;
         union
         {  struct
            {  int Handle,                     
                   __pad; } Handle;            
            struct
            {  int Volume,                     
                   DirEntry; } Entry;          
         } Is;
         int Size,
             Position,                         
             Access;                           
         short Length;
         int BytesToCopy;
         union
         {  short Value;
            int FullType[2]; } Contents;
      } fn_56_02;
      struct
      {  char SubFunction;
         short Flags;
         union
         {  struct
            {  int Handle,
                   __pad; } Handle;
            struct
            {  int Volume,
                   DirEntry; } Entry;
         } Is;
         int Position,
             Length;
      } fn_56_03;
      struct
      {  char SubFunction;
      } fn_57;
      struct
      {  char SubFunction,
              NameSpace,
              Mode;
         short SearchAttributes;
         int RIM,
             CreateAttributes;
         short AccessRights;
         char Volume;
         int Directory;
         char HandleFlag;
         int PathBytes;
         char PathComponents,
              Path[1];
      } fn_57_01;
      struct
      {  char SubFunction,
              NameSpace,
              Volume;
         int Directory;
         char HandleFlag;
         int PathBytes;
         char PathComponents,
              Path[1];
      } fn_57_02;
      struct
      {  char SubFunction,
              NameSpace;
         short SearchAttributes;
         int RIM;
         char Sequence[9];
      } fn_57_03;
      struct
      {  char SubFunction,
              NameSpace,
              Flags;
         short SearchAttributes;
         char SourceVolume;
         int SourceDirectory;
         char SourceHandleFlag,
              SourcePathComponents;
         char DestVolume;
         int DestDirectory;
         char DestHandleFlag,
              DestPathComponents;

         int SourcePathBytes;
         char *SourcePath;
         int DestPathBytes;
         char *DestPath;
      } fn_57_04;
      struct
      {  char SubFunction,
              NameSpace,
              DestNameSpace;
         short SearchAttributes;
         int RIM;
         char Volume;
         int Directory;
         char HandleFlag;
         int PathBytes;
         char PathComponents,
              Path[1];
      } fn_57_06;
      struct
      {  char SubFunction,
              NameSpace;
         short SearchAttributes;
         int InfoMask;
         struct nw_modify_dos_info
         {
            int attributes;
            short creationDate,
                  creationTime;
            int creatorID;
            short modifyDate,
                  modifyTime;
            int modifierID;
            short archiveDate,
                  archiveTime;
            int archiverID;
            short lastAccessDate,
                  inheritanceGrantMask,
                  inheritanceRevokeMask;
            int maximumSpace;
         } Info;
         char Volume;
         int Directory;
         char HandleFlag;
         int PathBytes;
         char PathComponents,
              Path[1];
      } fn_57_07;
      struct
      {  char SubFunction,
              NameSpace;
         short SearchAttributes;
         char Volume;
         int Directory;
         char HandleFlag;
         int PathBytes;
         char PathComponents,
              Path[1];
      } fn_57_08;
      struct
      {  char SubFunction,
              NameSpace,
              Volume;
         int Directory;
         char HandleFlag;
         int PathBytes;
         char PathComponents,
              Path[256];
      } fn_57_16;
      struct
      {  char SubFunction,
              Volume;
      } fn_57_18;
   } payload;
};

#define REQ_PREFIX   "W5be"
#define REQ_16       "Wb"
#define REQ_17       "Wb"
#define REQ_57       "b"
#define REQ_PATH     "blbn*b"
#define REQ_MODIFY   "lwwlwwlwwl3wl"

struct stdreply
{  struct ncp_reply_header header;
   union
   {  char Data[1536];
      struct
      {  unsigned short SectorsPerBlock,
                        TotalBlocks,
                        FreeBlocks,
                        TotalDirEntries,
                        FreeDirEntries;
         char Name[16];
         short Removable;
      } fn_12;
      struct
      {  int TotalBlocks,
             FreeBlocks,
             PurgableBlocks,
             NonPurgableBlocks,
             TotalDirEntries,
             FreeDirEntries;
         unsigned char SectorsPerBlock;
         char *Name;
      } fn_16_2c;
      struct
      {  short Count;
         int VolumeType,
             Flags,
             SectorSize,
             SectorsPerBlock,
             TotalBlocks,
             FreeBlocks,
             FreeSubAllocBlocks,
             FreeLimboSectors,
             LimboSectors,
             NonFreeSubAllocSectors,
             UnavailSubAllocSectors,
             TotalSubAllocBlocks,
             DataStreams,
             LimboDataStreams,
             OldestDeleted,
             CompressedStreams,
             CompressedLimboStreams,
             UncompressableStreams,
             PrecompressedSectors,
             CompressedSectors,
             MigratedFiles,
             MigratedSectors,
             FATBlocks,
             DirectoryBlocks,
             ExtDirBlocks,
             TotalDirEntries,
             FreeDirEntries,
             TotalDirExtants,
             FreeDirExtents,
             DefinedExtAttributes,
             UseddExtAttributes,
             ObjectID;
         short ModifyDate,
               ModifyTime;
         char *Name;
      } fn_16_33;
      struct volume_lookup
      {  int Count,
             VolumeID;
         char *Name;
      } fn_16_34;
      struct
      {  char Name[48];
         char Version,
               SubVersion,
               MaxConnections,
               CurrentConnections;
      } fn_17_11;
      struct
      {  char Key[8]; 
      } fn_17_17;
      struct user_lookup
      {  int UserID;
         short UserType;
         char Name[48];
         char LoginTime[7];
      } fn_17_1c;
      struct
      {  int ID;
         short Type;
         char Name[48];
      } fn_17_35;
      struct scan_object
      {  int ObjectID;
         short Type;
         char *Name;
         char Dynamic,
              Security,
              Properties;
      } fn_17_37;
      struct scan_property
      {  char *Name;
         char Flags,
              Security;
         int Sequence;
         char Value,
              More;
      } fn_17_3c;
      struct read_property
      {  char Value[128];
         char More,
              Flags;
      } fn_17_3d;
      struct
      {  struct queue_entry Entry;
      } fn_17_79;
      struct
      {  short Length;
      } fn_21;
      struct
      {  int Size;
      } fn_47;
      struct
      {  short Length;
         char Data[1];
      } fn_48;
      struct
      {  int Error,
             TotalLength,
             EAHandle;
      } fn_56_02;
      struct
      {  int Error,
             TotalLength,
             EAHandle,
             AccessFlag;
         short Length;
         union
         {  short Value;
            short Test[4];
            int FullType[2]; } Contents;
      } fn_56_03;
      struct
      {  int Handle;
         char Action;
      } fn_57_01;
      struct
      {  char Sequence[9];
         char *Name;
      } fn_57_02;
      struct
      {  char Sequence[9];
         char *Name;
      } fn_57_03;
      struct
      {  int SpaceAlloc;
         int Attributes;
         short Flags;
         int DataStreamSize;
         int TotalStreamSize;
         short NumberOfStreams;
         short CreationTime;
         short CreationDate;
         int CreatorID;
         short ModifyTime;
         short ModifyDate;
         int ModifierID;
         short LastAccessDate;
         short ArchiveTime;
         short ArchiveDate;
         int ArchiverID;
         short InheritedRightsMask;
         int DirEntNum;
         int DosDirNum;
         int VolNumber;
         int EADataSize;
         int EAKeyCount;
         int EAKeySize;
         int NSCreator;
         char NameLength;
         char Name[1];
      } fn_57_06;
      struct
      {  int Directory,
             DosDirectory;
         char Volume;
      } fn_57_16;
      struct
      {  char Count;
         char NameSpaces[1];
      } fn_57_18;
   } payload;
};


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Static data.                                                         |
   |                                                                        |
   +------------------------------------------------------------------------+ */

struct ExtensionMap
{  int Type;
   char *Name;
};

struct RiscIPX;
struct NWClient
{
   int ReturnStruct[8];

   struct RiscIPX *ri;

   LIST Mounts;
   LIST Servers;

   char NumServers,
        Flags,
      #define NWC_BINDERY        (1<<0)
      #define NWC_SET_CALLBACK   (1<<1)
      #define NWC_GET_MESSAGE    (1<<2)
      #define NWC_DONT_STREAM    (1<<7)
        Retries;              /* number of retries of Timeout length */

   int errno,
       TimeOffset,
       ReplyLength;

   struct ExtensionMap *ExtensionMap;
   struct Mount *CurrentMount;

   struct NWPrintSettings
   {  char *Banner,
           *Form,
           *Name,
           *Printer,
           *Queue,
           *Server,
           Copies,
           Tabs,
           FF,
           Notify,
           Lines,
           Columns;
   } PrintDefaults;

   struct stdreply reply;
   char SmallBuffer[512];
};

/* -------------------------------------------------------------------------- */

struct Server
{  NODE;

   struct NWClient *nw;

   struct Socket *NcpSocket,
                 *WdogSocket,
                 *MsgSocket;

   struct Address Address;

   LIST Files;    /* files opened with mt=NULL */

   char Flags,
      #define SERVER_LOGGED_IN      (1<<0)
      #define SERVER_GET_MESSAGE    (1<<1)
      #define SERVER_BIND_WDOG      (1<<2)
      #define SERVER_TCP_IP         (1<<3)
        ConnLow,
        ConnHigh,
        Timeout;              /* in 10ms units, no more than 2550ms */

   int BlockSize;
   char *ServerName,
        *UserName;
         
   struct stdrequest request;
};

/* -------------------------------------------------------------------------- */

struct Mount
{  NODE;

   LIST Files;    /* files opened for this mount point */
   
   struct Server *Server;

   char *LocalName,
        *RemoteName;

   int Directory,
       DosDirectory;

   int ScanItem,           /* last item scanned */
       ScanDirNumber,      /* namespace directory ID for directory we're scanning */
       ScanDosDirNumber;   /* DOS directory ID for directory we're scanning */
   char *ScanDirName;      /* current directory being scanned */
   LIST ScanDirCache;      /* list of block holding current directory contents */
   
   char Flags,
      #define MOUNT_USER0     (1<<0)   /* two user flags we don't care about, just pass around */
      #define MOUNT_USER1     (1<<1)
      #define MOUNT_USERFLAGS (MOUNT_USER0|MOUNT_USER1)
      #define MOUNT_LONGNAMES (1<<2)   /* equates to 4, the number of the OS/2 name space */
      #define MOUNT_RESCANDIR (1<<6)   /* must re-scan directory because file created/deleted */
      #define MOUNT_AUTOMOUNT (1<<7)   /* magic flag for SYS:LOGIN mount before login */
        Volume,
        MountCount,
        Sequence[10];
};


/* -------------------------------------------------------------------------- */

#define FILENAME_MAX 256

struct File
{  NODE;
   
   struct NWClient *nw;
   struct Mount *Mount;
   struct Server *Server;
   
   int RosHandle,
       CurrentSize,
       DirEntNum,
       VolNumber;

   union
   {  struct
      {  int Queue,
             Job; 
      } PrintJob;
      struct
      {  int DateLo,
             DateHi; 
      } File;
   } Is;

   char Flags,
      #define FILE_WRITABLE (1<<0)
      #define FILE_PRINTJOB (1<<1)
      #define FILE_GOTDATE  (1<<2)
        Action,         /* set to indicate how file was opened */
        Handle[6];      /* NetWare file handle (opaque) */

   char Name[1];
};


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Directory cache.                                                     |
   |                                                                        |
   +------------------------------------------------------------------------+ */

struct DirEntry
{  
   int Attributes,
       DataStreamSize,
       DirEntNum,
       VolNumber,
       FullType[2];
   short ModifyTime,
         ModifyDate,
         NameOffset,
         _pad;
};

struct DirCache
{  NODE;
   
   short SpaceLeft;
   char NumEntries;

   struct DirEntry Entries[1];
};

#define CACHE_BLOCKSIZE 4096


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   NetWare NCP constants.                                               |
   |                                                                        |
   +------------------------------------------------------------------------+ */

struct nw_entry_info
{
  int spaceAlloc;
  int attributes;
  short flags;
  int dataStreamSize;
  int totalStreamSize;
  short numberOfStreams;
  short creationTime;
  short creationDate;
  int creatorID;
  short modifyTime;
  short modifyDate;
  int modifierID;
  short lastAccessDate;
  short archiveTime;
  short archiveDate;
  int archiverID;
  short inheritedRightsMask;
  int dirEntNum;
  int DosDirNum;
  int volNumber;
  int EADataSize;
  int EAKeyCount;
  int EAKeySize;
  int NSCreator;
  char nameLength;
  char entryName[256];
};
#define ENTRY_INFO "llwllwwwlwwlwwwlwlllllllc"
#define ENTRY_INFO_NO_NAME "llwllwwwlwwlwwwlwlllllll"

typedef struct _MODIFY_DOS_INFO
{
  int   attributes;
  short createDate;
  short createTime;
  int   creatorID;
  short modifyDate;
  short modifyTime;
  int   modifierID;
  short archiveDate;
  short archiveTime;
  int   archiverID;
  short lastAccessDate;
  short inheritanceGrantMask;
  short inheritanceRevokeMask;
  int   maximumSpace;
} MODIFY_DOS_INFO;

typedef struct
{
  char  volNumber;
  int dirNumber;
  int searchDirNumber;
} SEARCH_SEQUENCE;

typedef struct
{
  char  openCreateMode;
  short searchAttributes;
  int reserved;
  int createAttributes;
  short accessRights;
  int NetWareHandle;
  char  openCreateAction;
} NW_NS_OPENCREATE, NW_NS_OPEN;


/* open/create modes */
#define OC_MODE_OPEN      0x01
#define OC_MODE_TRUNCATE  0x02
#define OC_MODE_REPLACE   0x02
#define OC_MODE_CREATE    0x08

/* open/create results */
#define OC_ACTION_NONE     0x00
#define OC_ACTION_OPEN     0x01
#define OC_ACTION_CREATE   0x02
#define OC_ACTION_TRUNCATE 0x04
#define OC_ACTION_REPLACE  0x04

/* return info mask */
#define RIM_NAME               0x0001L
#define RIM_ENTRY_NAME         0x0001L
#define RIM_SPACE_ALLOCATED    0x0002L
#define RIM_ATTRIBUTES         0x0004L
#define RIM_SIZE               0x0008L
#define RIM_TOTAL_SIZE         0x0010L
#define RIM_EA                 0x0020L
#define RIM_ARCHIVE            0x0040L
#define RIM_MODIFY             0x0080L
#define RIM_CREATION           0x0100L
#define RIM_OWNING_NAMESPACE   0x0200L
#define RIM_DIRECTORY          0x0400L
#define RIM_RIGHTS             0x0800L
#define RIM_ALMOST_ALL         0x0FEDL
#define RIM_ALL                0x0FFFL
#define RIM_REFERENCE_ID       0x1000L
#define RIM_NS_ATTRIBUTES      0x2000L
#define RIM_DATASTREAM_SIZES   0x4000L
#define RIM_COMPRESSED_INFO    0x80000000L
#define RIM_NS_SPECIFIC_INFO   0x80000000L

/* access rights attributes */
#define AR_READ            0x0001
#define AR_WRITE           0x0002
#define AR_READ_ONLY       0x0001
#define AR_WRITE_ONLY      0x0002
#define AR_DENY_READ       0x0004
#define AR_DENY_WRITE      0x0008
#define AR_COMPATIBILITY   0x0010
#define AR_WRITE_THROUGH   0x0040
#define AR_OPEN_COMPRESSED 0x0100

/* Trustee Access Rights in a network directory */
/* NOTE: TA_OPEN is obsolete in 3.x */
#ifndef TA_NONE
#define TA_NONE       0x00
#define TA_READ       0x01
#define TA_WRITE      0x02
#define TA_OPEN       0x04
#define TA_CREATE     0x08
#define TA_DELETE     0x10
#define TA_OWNERSHIP  0x20
#define TA_SEARCH     0x40
#define TA_MODIFY     0x80
#define TA_ALL        0xFB
#endif

#define ATTR_READONLY         (1L<<0)
#define ATTR_HIDDEN           (1L<<1)
#define ATTR_SYSTEM           (1L<<2)
#define ATTR_EXECUTE          (1L<<3)
#define ATTR_DIR              (1L<<4)
#define ATTR_ARCH             (1L<<5)
#define ATTR_SHARED           (1L<<7)
#define ATTR_DONTSUBALLOCATE  (1L<<11)
#define ATTR_TRANSACTIONAL    (1L<<12)
#define ATTR_PURGE            (1L<<16)
#define ATTR_RENAMEINHIBIT    (1L<<17)
#define ATTR_DELETEINHIBIT    (1L<<18)
#define ATTR_DONTCOMPRESS     (1L<<27)

#define NCP_MAX_SYMLINK_SIZE  256   /* 512 for linux, 256 for RiscOS */

/* search attributes */
#ifndef SA_HIDDEN
#define SA_NORMAL         0x0000
#define SA_HIDDEN         0x0002
#define SA_SYSTEM         0x0004
#define SA_SUBDIR_ONLY    0x0010
#define SA_SUBDIR_FILES   0x8000
#define SA_ALL            0x8006
#endif

#define NW_TYPE_FILE      0x8000
#define NW_TYPE_SUBDIR    0x0010

#define NW_NAME_CONVERT     0x03
#define NW_NO_NAME_CONVERT  0x04

/* modify mask - use with MODIFY_DOS_INFO structure */
#define DM_ATTRIBUTES             0x0002L
#define DM_CREATE_DATE            0x0004L
#define DM_CREATE_TIME            0x0008L
#define DM_CREATOR_ID             0x0010L
#define DM_ARCHIVE_DATE           0x0020L
#define DM_ARCHIVE_TIME           0x0040L
#define DM_ARCHIVER_ID            0x0080L
#define DM_MODIFY_DATE            0x0100L
#define DM_MODIFY_TIME            0x0200L
#define DM_MODIFIER_ID            0x0400L
#define DM_LAST_ACCESS_DATE       0x0800L
#define DM_INHERITED_RIGHTS_MASK  0x1000L
#define DM_MAXIMUM_SPACE          0x2000L

struct nw_property
{
   char  Value[128],
         More_flag,
         Property_flag;
};

   
/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Function prototypes.                                                 |
   |                                                                        |
   +------------------------------------------------------------------------+ */

/* ----- ncp.c -------------------------------------------------------------- */

extern struct NWClient *nwclient_init(register struct RiscIPX *ri);
extern int nwclient_exit(register struct NWClient *nw);

int _ncp_request(char *requestFormat,char *replyFormat,char *strings,register struct Server *fs);
#define ncp_request(reqf,rplf,str,fs) _ncp_request(REQ_PREFIX reqf,rplf,str,fs)

extern int wdog_handler(struct mbuf *mb,struct UnpackedHeader *header,void *fsptr);
extern int make_path(char *nwpath,const char *name);
extern int nwclient_init_search(struct Mount *mt,const char *directory,char Sequence[10],struct NWClient *nw);
extern int nwclient_search(struct Mount *mt,char *name);

/* ----- nwswis.c ----------------------------------------------------------- */

extern int *nwclient_attach(char *tree,int mode,register struct NWClient *nw);
   #define ATTACH_TREE   0
   #define ATTACH_SERVER 1
   #define ATTACH_NODE   2
   #define ATTACH_TCPIP  3
extern int nwclient_detach(struct Server *fs,register struct NWClient *nw);
extern int nwclient_login(const char *name,char *password,register struct NWClient *nw);
extern int nwclient_logout(struct Server *fs,register struct NWClient *nw);

extern int nwclient_dismount(struct Mount *mt,struct NWClient *nw);
extern int *nwclient_mount(const char *localName,const char *remoteName,int flags,register struct NWClient *nw);
extern int nwclient_setpass(const char *user,char *oldpass,char *newpass,register struct NWClient *nw);
extern int *nwclient_mountlist(struct Mount *mt,register struct NWClient *nw);
extern int nwclient_systime(struct Server *fs,register struct NWClient *nw);

/* ----- fileio.c ----------------------------------------------------------- */

extern struct File *nwclient_open(struct Mount *mt,const char *name,int mode,register struct NWClient *nw);
   #define O_OPEN          OC_MODE_OPEN
   #define O_REPLACE       OC_MODE_REPLACE
   #define O_CREATE        OC_MODE_CREATE
   #define O_RDWR          ((AR_READ|AR_WRITE)<<8)
   #define O_READ          (AR_READ<<8)
   #define O_WRITE         (AR_WRITE<<8)
   #define O_DENYREAD      (AR_DENY_READ<<8)
   #define O_DENYWRITE     (AR_DENY_WRITE<<8)
   #define O_COMPATIBILITY (AR_COMPATIBILITY<<8)
   #define O_WRITETHROUGH  (AR_WRITE_THROUGH<<8)
   #define O_COMPRESSED    (AR_OPEN_COMPRESSED<<8)
extern int nwclient_close(struct File *file,struct NWClient *nw);
extern int nwclient_read(struct File *file,void *buffer,int offset,int length,struct NWClient *nw);
extern int nwclient_write(struct File *file,const void *buffer,int offset,int length,struct NWClient *nw);
extern int nwclient_filesize(struct File *file,struct NWClient *nw);
extern struct File *nwclient_spool(struct NWPrintSettings *Settings,register struct NWClient *nw);

/* ----- nwnds.c ------------------------------------------------------------ */

struct nwpropflags
{  char Flags,
        Security,
        Value;
};

struct ncp_bindery_object
{  long object_id;
   short object_type;
   char object_name[48]; 
};

extern int ncp_get_bindery_object_id(register struct Server *fs,int type,const char *name,struct ncp_bindery_object *target);
extern int ncp_login_user(register struct Server *fs,const char *username,const char *password);
extern void shuffle(const unsigned char *lon,const unsigned char *buf,int buflen,unsigned char *target);
extern void nw_encrypt(const unsigned char *fra,const unsigned char *buf,unsigned char *til);
extern void newpassencrypt(char *old, char *new, char *out);
extern int ncp_get_encryption_key(register struct Server *fs,char *target);
extern int nwclient_scanobject(int objectType,const char *objectName,int sequence,
                               struct scan_object **result,struct Server *fs);
extern int nwclient_scanprop(int objectType,const char *objectName,int sequence,
                      struct scan_property **result,struct Server *fs);
extern int nwclient_readprop(int objectType,const char *objectName,const char *propName,
                      int segment,struct read_property **result,
                      struct Server *fs);

/* ----- nwutil.c ----------------------------------------------------------- */

extern char *strdup(const char *string);
extern void nwclient_logmessage(char *message);
extern struct Server *lookup_server(const char *Name,const char **Tail,struct NWClient *nw);

extern struct Mount *RosToNWName(char *buffer,const char *rosName,struct NWClient *nw,int dirFlag);
extern void NWToRosName(struct Mount *mt,char *buffer,const char *source);
extern int RosAttrib(int Attributes);
extern void NWToRosDate(int *Result,int Date,int Time,int Offset);
extern void RosToNWDate(int DateHi,int DateLo,short *Date,short *Time,int Offset);
extern int InventFileType(register struct NWClient *nw,const char *name);


/* ----- fsglue.arm --------------------------------------------------------- */

extern void AddFilingSystem(struct NWClient *nw);
extern void RemoveFilingSystem(struct NWClient *nw);
extern void RegisterFree(struct NWClient *nw);
extern void DeregisterFree(struct NWClient *nw);
extern void AddCallBack(struct NWClient *nw);
extern void RemoveCallBack(struct NWClient *nw);

extern int mul100(int block[2],int value);
extern int Version;


/* ----- fsmain.c ----------------------------------------------------------- */

void ncpfs_init(register struct NWClient *nw);
void ncpfs_exit(register struct NWClient *nw);

extern int ncp_connection_info(struct user_lookup **user,int connection,
                               struct Server *fs,register struct NWClient *nw);

#define SOFTLINK 0xfdc
#define UNIXEXEC 0xfe6
#define SYMLNK_MAGIC0       0x6c6d7973     /* "symlnk->" */
#define SYMLNK_MAGIC1       0x3e2d6b6e

/* ----- fsdirs.c ----------------------------------------------------------- */

extern void FlushDirCache(struct Mount *mt,struct NWClient *nw);

/* ----- stars.c ------------------------------------------------------------ */

extern void DefaultExtensions(register struct NWClient *nw);
extern struct kernel_error *StarDetach(const char *commandTail,int numParms,struct NWClient *nw);


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Errors.                                                              |
   |                                                                        |
   +------------------------------------------------------------------------+ */

extern struct kernel_error *LookupError(int error);
extern int ReportError(struct kernel_error *err,int flags,char *name);
extern struct kernel_error Error_WaitingNCPReq,
                           Error_DelCurrent,
                           Error_VolName,
                           Error_BadParms,
                           Error_DelRoot,
                           Error_BadRename,
                           Error_Access,
                           Error_FileOpen,
                           Error_Locked,
                           Error_Exists,
                           Error_DiskFull,
                           Error_ReadError,
                           Error_WriteError,
                           Error_InvalidName,
                           Error_NotFound,
                           Error_NoMemory;


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Debugging.                                                           |
   |                                                                        |
   +------------------------------------------------------------------------+ */

#ifdef DEBUG
extern void ShowError(char *,int);
#else
#define ShowError(i) do {} while(0)
#endif


/* ----- EOF NWCLIENT.H ----- */


