/* #------------------------------------------------------------------------#
   |                                                                        |
   |   IPX.C                                                                |
   |                                                                        |
   |   IPX socket layer for RiscOS.                                         |
   |                                                                        |
   |   Copyright 1998, Frank A. Vorstenbosch.                               |
   |                                                                        |
   #------------------------------------------------------------------------# */

/* $Id: ipx.c,v 0.20 2001/12/22 13:46:51 frank Exp $ */

#include "iriscipx.h"

#define VERSION 0x00042

#pragma On(Check_ltorg)

/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Global variables.                                                    |
   |                                                                        |
   +------------------------------------------------------------------------+ */

const struct Address BroadcastAddress =
{  0L,"\xff\xff\xff\xff\xff\xff",0 };
static const char BroadcastNode[IPX_NODE_LEN]="\xff\xff\xff\xff\xff\xff";

#define IPXHDRFMT "2Wzb2A"
static const char IPX8022Format[]="2k\xe0k\003" IPXHDRFMT;   /* for IPX over 802.2 */
static const char IPX8023Format[]=IPXHDRFMT;            /* for IPX over 802.3 or Ethernet II */
static const char IPXSNAPFormat[]="2k\xaak\0033k\0k\x81k\x37" IPXHDRFMT; /* for IPX over Ethernet SNAP */

static const char *Format[6]={ NULL,IPX8022Format,IPX8023Format,IPXSNAPFormat,IPX8023Format,IPX8023Format };

/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Interrupt routine.                                                   |
   |                                                                        |
   +------------------------------------------------------------------------+ */

#if defined DEBUG && (DEBUG&64 || DEBUG&128)
static void DumpPacket(struct mbuf *mb,char *title,char *mac,int bytes,int frame);
#endif

#if defined DEBUG && DEBUG&256
static void ShowMbuf(struct mbuf *mb,const char *title)
{
   Trace("\n ");
   Trace(title);
   Trace("={\n {  struct mbuf *next=");    TraceVal((int)(mb->next));
   Trace(",\n                *list=");     TraceVal((int)(mb->list));
   Trace(";\n    int off=");               TraceVal(mb->off);
   Trace(",\n        len=");               TraceVal(mb->len);
   Trace(",\n        inioff=");            TraceVal(mb->inioff);
   Trace(",\n        inilen=");            TraceVal(mb->inilen);
   Trace(";\n    unsigned char type=");    TraceByte(mb->type);
   Trace(",\n                  sys1=");    TraceByte(mb->sys1);
   Trace(",\n                  sys2=");    TraceByte(mb->sys2);
   Trace(",\n                  flags=");   TraceByte(mb->flags);
   Trace(";\n    struct pkthdr"
          "\n    {  int len=");            TraceVal(mb->pkthdr.len);
   Trace(";\n       void *rcv_if=");       TraceVal((int)(mb->pkthdr.rcv_if));
   Trace(";\n    } pkthdr;\n };");
}
#endif


void ReceiveHandler(void *ri_ptr,struct mbuf *mb,int ft)
{  register struct Socket *sk;
   struct UnpackedHeader header;
   register struct RiscIPX *ri=ri_ptr;
   register int i;

   if(!ft)
      ft=ri->FrameType;

   #if defined DEBUG && DEBUG&2
   Trace("\n ReceiveHandler(,");
   TraceVal((int)mb);
   Trace(",");
   TraceVal(ft);
   Trace(",)");
   #endif

   if(!ri->DisableCount)
   {  ri->DisableCount=1;

      #if defined DEBUG && DEBUG&256
      ShowMbuf(mb,"mb");
      ShowMbuf(mb->next,"mb->next");
      #endif

      if(mb->len==32 && (
                         #ifdef ECONET
                         ft==FRAMETYPE_ECONET || 
                         #endif
                         ft==FRAMETYPE_E2 || ft==FRAMETYPE_8023 || 
                         (i=Unpack(mb->next,&header,"W",NULL))>0))
      {  
         #if defined DEBUG && DEBUG&2
         Trace("\n len=32");
         #endif

         switch(ft)
         {  default:
            case FRAMETYPE_AUTO:
               #if defined DEBUG && DEBUG&4
               Trace(" Autodetect");
               #endif
               switch(header.CheckSum)
               {  case 0xe0e0:
                     #if defined DEBUG && (DEBUG&128)
                     DumpPacket(mb->next,"\n  recv_int from ",(char *)mb+mb->off+8,ri->mbctl.count_bytes(&ri->mbctl,mb->next),FRAMETYPE_8022);
                     #endif
                     ft=FRAMETYPE_8022;
                     goto its8022;
                  case 0xffff:
                     #if defined DEBUG && (DEBUG&128)
                     DumpPacket(mb->next,"\n  recv_int from ",(char *)mb+mb->off+8,ri->mbctl.count_bytes(&ri->mbctl,mb->next),FRAMETYPE_8023);
                     #endif
                     ft=FRAMETYPE_8023;
                     goto its8023;
                  #if 0
                  case 0x8137:
                     goto itse2;
                  #endif
                  case 0xaaaa:
                     #if defined DEBUG && (DEBUG&128)
                     DumpPacket(mb->next,"\n  recv_int from ",(char *)mb+mb->off+8,ri->mbctl.count_bytes(&ri->mbctl,mb->next),FRAMETYPE_SNAP);
                     #endif
                     ft=FRAMETYPE_SNAP;
                     goto trysnap;
               }
               #if defined DEBUG && (DEBUG&128)
               DumpPacket(mb->next,"\n  recv_int from ",(char *)mb+mb->off+8,ri->mbctl.count_bytes(&ri->mbctl,mb->next),FRAMETYPE_AUTO);
               #endif
               goto eat;

            case FRAMETYPE_8022:
               #if defined DEBUG && (DEBUG&128)
               DumpPacket(mb->next,"\n  recv_int from ",(char *)mb+mb->off+8,ri->mbctl.count_bytes(&ri->mbctl,mb->next),FRAMETYPE_8022);
               #endif
               if(header.CheckSum!=0xe0e0)
                  goto eat;
       its8022:mb->pkthdr.len=3;
               break;

            #ifdef ECONET
            case FRAMETYPE_ECONET:
            #endif
            case FRAMETYPE_E2:
            case FRAMETYPE_8023:
               #if defined DEBUG && (DEBUG&128)
               DumpPacket(mb->next,"\n  recv_int from ",(char *)mb+mb->off+8,ri->mbctl.count_bytes(&ri->mbctl,mb->next),FRAMETYPE_8023);
               #endif
               #if 0
               if(header.CheckSum!=0xffff)
                  goto eat;
               #endif
       its8023:mb->pkthdr.len=0;
               break;

            case FRAMETYPE_SNAP:
               #if defined DEBUG && (DEBUG&128)
               DumpPacket(mb->next,"\n  recv_int from ",(char *)mb+mb->off+8,ri->mbctl.count_bytes(&ri->mbctl,mb->next),FRAMETYPE_SNAP);
               #endif
               if(header.CheckSum!=0xaaaa)
                  goto eat;
       trysnap:
               mb->pkthdr.len=8;
               break;
         }

         if((i=UnpackN(mb->next,&header,IPX8023Format,NULL,mb->pkthdr.len))<=0)
            goto eat;
         mb->pkthdr.len+=i;

         #if defined DEBUG && (DEBUG&128)
         Trace(" PT=");
         TraceByte(header.PacketType);
         Trace(" port=");
         TraceVal(header.Destination.Port);
         Trace(" net=");
         TraceVal(header.Destination.Network);
         #endif

         forList(&ri->Sockets,sk)
         {  
            #if defined DEBUG && (DEBUG&128)
               Trace("\n sk->flags");
               TraceVal(sk->Flags);
               if(sk->Flags&SOCKET_CONNECT_PORT)
                  Trace(" port");
               if(sk->Flags&SOCKET_CONNECT_NETWORK)
                  Trace(" net");
               if(sk->Flags&SOCKET_CONNECT_NODE)
                  Trace(" node");

               Trace("\n sk->acceptpt");
               if(sk->Flags&SOCKET_ANY_TYPE)
                  Trace(" any");
               else
                  TraceVal(sk->AcceptPT);

               Trace("\n sk->port");
               TraceVal(sk->Port);

               Trace("\n sk->con");
               TraceVal(((int *)&sk->Connection)[0]);
               TraceVal(((int *)&sk->Connection)[1]);
               TraceVal(((int *)&sk->Connection)[2]);

               Trace(" ifs(");
               if((sk->Flags&SOCKET_BOUND))  Trace("1");
               if((sk->Port==header.Destination.Port)) Trace("2");
               if(((sk->AcceptPT==header.PacketType) || (sk->Flags&SOCKET_ANY_TYPE))) Trace("3");
               if((!(sk->Flags&SOCKET_CONNECT_PORT) || sk->Connection.Port==header.Source.Port)) Trace("4");
               if((!(sk->Flags&SOCKET_CONNECT_NETWORK) || sk->Connection.Network==header.Source.Network)) Trace("5");
               if((!(sk->Flags&SOCKET_CONNECT_NODE) || !memcmp(sk->Connection.Node,header.Source.Node,IPX_NODE_LEN))) Trace("6");
               Trace(")");
            #endif

            if((sk->Flags&SOCKET_BOUND) &&
               (sk->Port==header.Destination.Port) &&
               ((sk->AcceptPT==header.PacketType) || (sk->Flags&SOCKET_ANY_TYPE)) &&
               (!(sk->Flags&SOCKET_CONNECT_PORT) || sk->Connection.Port==header.Source.Port) &&
               (!(sk->Flags&SOCKET_CONNECT_NETWORK) || sk->Connection.Network==header.Source.Network) &&
               (!(sk->Flags&SOCKET_CONNECT_NODE) || !memcmp(sk->Connection.Node,header.Source.Node,IPX_NODE_LEN)))
            {
               #if defined DEBUG && (DEBUG&64) && !(DEBUG&128)
               DumpPacket(mb->next,"\n  recv_int from ",(char *)mb+mb->off+8,ri->mbctl.count_bytes(&ri->mbctl,mb->next),ft);
               #endif

               mb->type=ft;
               if(!sk->Handler || sk->Handler(mb,&header,sk->HandlerData))
               {  
                  #if defined DEBUG && (DEBUG&128)
                  Trace("\n didn't eat it");
                  #endif

                  if(sk->TailQueue)
                     sk->TailQueue->list=mb;
                  else
                     sk->ReadQueue=mb;
                  sk->TailQueue=mb; }

               ri->DisableCount=0;
               return;
            }
         }
      }
      /* end if(mb->len==32) */
   }

eat:
   #if defined DEBUG && (DEBUG&128)
   Trace("\n ignore");
   #endif

   ri->mbctl.freem(&ri->mbctl,mb);
   ri->DisableCount=0;
}

/* -------------------------------------------------------------------------- */

#ifdef ECONET

static int EcoListen(register struct RiscIPX *ri)
{
   Trace(" EcoListen {");
   ri->RxCB=0;

   if((ri->Listen=ri->mbctl.alloc_s(&ri->mbctl,32,NULL))==NULL)
      return IPX_NOT_ENOUGH_MEMORY;

   #if defined DEBUG && DEBUG&256
   ShowMbuf(ri->Listen,"ri->Listen");
   #endif

   if((ri->Listen->next=ri->mbctl.alloc_s(&ri->mbctl,1024+ECONET_PREHEADER,NULL))==NULL)
   {  
abort:ri->mbctl.freem(&ri->mbctl,ri->Listen);
      ri->Listen=NULL;
      Trace(" nomem }");
      return IPX_NOT_ENOUGH_MEMORY;
   }

   #if defined DEBUG && DEBUG&256
   ShowMbuf(ri->Listen->next,"ri->Listen->next");
   #endif

   if((ri->RxCB=Econet_CreateReceive(ECONET_IPX_PORT,0,0,mtod(ri->Listen->next,char *)+ECONET_PREHEADER,1024))==0)
   {  ri->mbctl.freem(&ri->mbctl,ri->Listen->next);
      goto abort; }

   Trace(" OK }");
   return 0;
}

/* .......................................................................... */

void EconetHandler(register struct RiscIPX *ri,int handle,int status,int port)
{  int flag,station,network,length;
   struct mbuf *mb;
   char *buffer;
   struct rx_header
   {  int z0,z1,
          srchi,srclo;
      char dst[6],_pad[2];
      int type,
          error; } *rxh;
         
   #if defined DEBUG
   Trace("\n EconetHandler(");
   TraceVal((int)ri);
   TraceVal(handle);
   TraceVal(port);
   Trace(") {");
   #endif

   /* ----- build the mbufs that the ReceiveHandler function expects ----------- */

   status=Econet_ReadReceive(handle,&flag,&port,&station,&network,(void **)&buffer,&length);

   Trace("\n flag=");   TraceByte(flag);
   Trace(",port=");     TraceByte(port);
   Trace(",station=");  TraceByte(station);
   Trace(",network=");  TraceByte(network);
   Trace(",address=");  TraceVal((int)buffer);
   Trace(",length=");   TraceVal(length);

   if(status!=9)
   {  Trace(" !=9 }");
      return; }

   mb=ri->Listen;
   EcoListen(ri);

   #if defined DEBUG && DEBUG&256
   ShowMbuf(mb,"mb");
   if(mb->next)
      ShowMbuf(mb->next,"mb->next");
   #endif

   rxh=mtod(mb,struct rx_header *);
   rxh->srchi=rxh->z0=rxh->z1=rxh->error=0;
   rxh->srclo=station<<8;
   memcpy_node(rxh->dst,ri->EconetNode);

   buffer=mtod(mb->next,char *);
   length+=ECONET_PREHEADER;
   *buffer++=-1;              /* IPX checksum */
   *buffer++=-1;
   *buffer++=length>>8;       /* payload length */
   *buffer++=length;
   *buffer++=0;               /* hop count */
   *buffer++=flag;            /* packet type */
   rxh->type=length;

   TraceMem(mtod(mb,char *),32);
   TraceMem(mtod(mb->next,char *),64);

   ri->mbctl.trim(&ri->mbctl,mb->next,length-(1024+ECONET_PREHEADER),NULL);

   ReceiveHandler((void *)ri,mb,FRAMETYPE_ECONET);
   
   Trace(" }");
}
#endif


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Open and close functions.                                            |
   |                                                                        |
   +------------------------------------------------------------------------+ */

static int MatchFrame(char *buffer);

_Asm int RandIPXPort(void)
{
   swi   0x042    ; OS_ReadMonotonicTime
   add   %r0,%r0,%r0,LSR #8
   and   %r0,%r0,#255<<4
};

#ifdef ECONET
_Asm int Econet_ReadLocalStationAndNet(void)
{
   swi   0x6000a  ; Econet_ReadLocalStationAndNet
   mvnvs %r0,#0
   orrvc %r0,%r0,%r1,LSL#8
}

_Asm int Econet_ClaimPort(int port) 
{  % con port
   mov   %r0,#port
   swi   0x60015  ; Econet_ClaimPort
   mvnvs %r0,#0
   movvc %r0,#0
}

_Asm void Econet_ReleasePort(int port) 
{  % con port
   mov   %r0,#port
   swi   0x60012  ; Econet_ReleasePort
}

_Asm int Econet_AbandonReceive(int handle)
{  % reg handle
   mov   %r0,handle
   swi   0x60003  ; Econet_AbandonReceive
}
#endif

struct RiscIPX *ipx_init(void)
{  struct dibchain
   {  struct dibchain *next;
      struct dib
      {  int swibase;
         char *name;
         int unit;
         char *node,
              *module,
              *location;
         int slot,
             inquire; } *interface; } *p;
   register struct RiscIPX *ri;
   struct Route *rt;
   int i;
   char buffer[20];

   if((ri=malloc(sizeof(struct RiscIPX)))==NULL)
      return NULL;

   memset((void *)ri,0,sizeof(struct RiscIPX));

   NewList(&ri->Sockets);

   #if defined DEBUG
   ri->NextPort=FIRSTPORT+RandIPXPort();
   #else
   ri->NextPort=FIRSTPORT;
   #endif

   ri->DisableCount=ri->Flags=0;
   ri->FrameType=FRAMETYPE_AUTO;
   ri->MTU=1600;

   if((i=ReadVarVal("RiscIPX$Frame",buffer,19,0))>0)
   {  buffer[i]=0;

      if((ri->FrameType=MatchFrame(buffer))==FRAMETYPE_AUTO)
      {  char *p=buffer;
         while(*p && *p!='_')
            p++;
         if(*p)
         {  p++;
            ri->FrameType=MatchFrame(p); }
      }
   }

   ri->mbctl.mbcsize  =sizeof(struct mbctl);
   ri->mbctl.mbcvers  =17;
   ri->mbctl.flags    =0;       /* give me supervisor mode entry points */
   ri->mbctl.advminubs=0;
   ri->mbctl.advmaxubs=1600;
   ri->mbctl.mincontig=0;
   ri->mbctl.spare1   =0;

   if(OpenSession(&ri->mbctl))
   {  
 bad: free(ri);
      return NULL; }

   if(ipxrtr_init(ri))
      goto bad;

   #ifdef ECONET
   if(ri->FrameType==FRAMETYPE_AUTO || ri->FrameType==FRAMETYPE_ECONET)
   {
      Trace("\n Init for Econet");
      
      i=Econet_ReadLocalStationAndNet();

      Trace(" station=");   TraceVal(i);

      if(i!=-1)
      {
         ri->EconetNode[5]=i; /* memset takes care of everything else */
         i=(i>>8)&0xff;

         if((rt=ipxrtr_add_route(0,NULL,ri))==NULL)
            goto bad;

         rt->Network=0xec000000|i;
         rt->Flags=FRAMETYPE_ECONET;   
         memcpy_node(rt->Node,ri->EconetNode);

         Trace(" claim");
         Claim(0x10,&EventHandler,ri);
         OSByte(14,14); /* enable event 14 */

         Econet_ClaimPort(ECONET_IPX_PORT);
         if(EcoListen(ri))
            goto bad;

         ri->SwiBase=0x40000;
         ri->MTU=1024;  /* fixed Econet MTU */
         ri->Flags|=IPX_REGISTERED_ECONET;
      }
      else
      {  if(ri->FrameType==FRAMETYPE_ECONET)
            goto bad;     /* "No network adaptor found." */
      }
   }

   if(ri->FrameType!=FRAMETYPE_ECONET)
   #endif
   {
      Trace("\n Init for Ethernet");

      p=ServiceCall9B();

      while(p)
      {  if((p->interface->module[0]=='E' || p->interface->module[0]=='e') &&
            (p->interface->module[1]=='T' || p->interface->module[1]=='t') &&
            (p->interface->module[2]=='H' || p->interface->module[2]=='h'))
            break;
         p=p->next;
      }

      if(p==NULL)
         return NULL;     /* "No network adaptor found." */

      PatchSwis(ri->SwiBase=p->interface->swibase);
      ri->Unit=p->interface->unit;
      memcpy(ri->Node,p->interface->node,IPX_NODE_LEN);  /* use slow copy because of alignment problems */
      i=GetMTU(0,ri->Unit);
      if(i<ri->MTU)
         ri->MTU=i;

      if(DCIVersion()<403)
         goto bad;

      if(ri->FrameType!=FRAMETYPE_E2)
      {  
         Trace("\n register 0x40000");
         if(Register(0,ri->Unit,0x40000,2,0,(void *)ri,&IntHandler))
         {  if(ri->FrameType!=FRAMETYPE_AUTO)
               goto bad;
            Trace("\nFall back onto Ethernet_II");
            ri->FrameType=FRAMETYPE_E2; }
         else
            ri->Flags|=IPX_REGISTERED_40000;
      }

      if((ri->FrameType==FRAMETYPE_E2 || ri->FrameType==FRAMETYPE_AUTO))
      {  
         Trace("\n register 0x18137");
         if(Register(0,ri->Unit,0x18137,2,0,(void *)ri,&IntHandlerE2))
            goto bad;
         else
            ri->Flags|=IPX_REGISTERED_18137;
      }

      if(ri->FrameType==FRAMETYPE_AUTO)
      {  for(i=FRAMETYPE_FIRST;i<=FRAMETYPE_LAST;i++)
         #ifdef ECONET
         if(i!=FRAMETYPE_ECONET)
         #endif
         {  if((rt=ipxrtr_add_route(0,NULL,ri))==NULL)
               goto bad;
            rt->Flags=i|ROUTE_UNKNOWN;
            memcpy_node(rt->Node,ri->Node);
      }  }
      else
      {  if((rt=ipxrtr_add_route(0,NULL,ri))==NULL)
            goto bad;
         rt->Flags=ri->FrameType|ROUTE_UNKNOWN;
         memcpy_node(rt->Node,ri->Node);
      }
   }

   Trace("\n try to get some routes...");
   (void)ipxrtr_lookup(0xffffffff,ri);    /* try to get some routes */

   return ri;
}

/* -------------------------------------------------------------------------- */

int ipx_exit(register struct RiscIPX *ri)
{  struct Socket *sk,*next;

   Trace("\n ipx_exit");
   ipxrtr_exit(ri);
   
   forList2(&ri->Sockets,sk,next)
      ipx_close(sk,ri);
   
   #ifdef ECONET
   if(ri->Flags&IPX_REGISTERED_ECONET)
   {  Trace(" deregister econet");
      if(ri->RxCB)
         Econet_AbandonReceive(ri->RxCB);

      if(ri->Listen)
         ri->mbctl.freem(&ri->mbctl,ri->Listen);

      Trace(" Release");
      Econet_ReleasePort(ECONET_IPX_PORT);
      OSByte(13,14); /* disable event 14 */
      Release(0x10,&EventHandler,ri);
   }
   #endif
   if(ri->Flags&IPX_REGISTERED_40000)
   {  Trace(" deregister 0x40000");
      Register(1,ri->Unit,0x40000,2,0,(void *)ri,&IntHandler); }
   if(ri->Flags&IPX_REGISTERED_18137)
   {  Trace(" deregister 0x18137");
      Register(1,ri->Unit,0x18137,2,0,(void *)ri,&IntHandlerE2); }

   CloseSession(&ri->mbctl);
   free(ri);

   return 0;
}

/* -------------------------------------------------------------------------- */

int ipx_sizeof_struct(void)
{
   return sizeof(struct RiscIPX);
}


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Create and destroy sockets.                                          |
   |                                                                        |
   +------------------------------------------------------------------------+ */

struct Socket *ipx_socket(register struct RiscIPX *ri)
{  struct Socket *sk;
   const char *p;

   if((sk=malloc(sizeof(struct Socket)))==NULL)
      return NULL;
   
   sk->SendPT=IPX_PT_USER;
   sk->Connection.Port=sk->Connection.Network=sk->Port=sk->Flags=0;
   sk->Handler=NULL;
   sk->ReadQueue=sk->TailQueue=NULL;
   memcpy_node(sk->Connection.Node,p=BroadcastNode);
   
   Disable(ri);
   AddTail(&ri->Sockets,sk);
   Enable(ri);

   return sk;
}

/* -------------------------------------------------------------------------- */

int ipx_close(struct Socket *sk,register struct RiscIPX *ri)
{  struct mbuf *mb,*next;

   if(!sk)
      return 0;

   Disable(ri);
   RemoveNode(sk);
   
   Enable(ri);

   mb=sk->ReadQueue;
   while(mb)
   {  next=mb->list;
      mb->list=NULL; /* paranoid */
      ri->mbctl.freem(&ri->mbctl,mb);
      mb=next; }
   
   free(sk);
   return 0;
}

/* -------------------------------------------------------------------------- */

int ipx_recv_handler(struct Socket *sk,
                     int (*Handler)(struct mbuf *,struct UnpackedHeader *,void *),
                     void *Data,
                     register struct RiscIPX *ri)
{
   sk->Handler=Handler;
   sk->HandlerData=Data;

   return 0;
}

/* -------------------------------------------------------------------------- */

int ipx_bind(struct Socket *sk,int packetType,int port,register struct RiscIPX *ri)
{  struct Socket *p;

   #ifdef DEBUG
   Trace("\n bind(");
   TraceVal((int)sk);
   TraceVal(packetType);
   TraceVal(port);
   Trace(") {");
   #endif

   if(!port)
   {
      forList(&ri->Sockets,p)
         if(/*(p->Flags&SOCKET_BOUND) &&*/ p->Port==ri->NextPort)
         {  ri->NextPort++;
            if(ri->NextPort>LASTPORT)
               ri->NextPort=FIRSTPORT;
            HeadNode(&ri->Sockets,p); }
      port=ri->NextPort++;
   }

   forList(&ri->Sockets,p)
      if((p->Flags&SOCKET_BOUND) && p->Port==port &&
         !(packetType && p->AcceptPT==packetType))
      {  Trace(" inuse }");
         return IPX_CANNOT_BIND; }

   #if defined DEBUG && (DEBUG&2)
   TraceVal(port);
   #endif

   if(!packetType)
      sk->Flags|=SOCKET_ANY_TYPE;
   else
      sk->AcceptPT=packetType;
   sk->Port=port;
   sk->Flags|=SOCKET_BOUND;

   Trace(" }");

   return 0;
}

/* -------------------------------------------------------------------------- */

int ipx_listen(struct Socket *sk,register struct RiscIPX *ri)
{  return IPX_UNIMPLEMENTED; }

/* -------------------------------------------------------------------------- */

int ipx_accept(struct Socket *sk,register struct RiscIPX *ri)
{  return IPX_UNIMPLEMENTED; }

/* -------------------------------------------------------------------------- */

int ipx_connect(struct Socket *sk,int packetType,const struct Address *name,register struct RiscIPX *ri)
{  
   #ifdef DEBUG
   Trace("\n connect ");
   #endif

   if(!name || (name->Network && !ipxrtr_lookup(name->Network,ri)))
      return IPX_NO_ROUTE_TO_NETWORK;

   if(packetType)
      sk->SendPT=packetType;
   else
      sk->SendPT=IPX_PT_USER;
   sk->Flags|=SOCKET_BOUND;

   #if 0
   Disable(ri);
   RemoveNode(sk);
   AddHead(&ri->Sockets,sk);
   Enable(ri);
   #endif

   memcpy_address(&sk->Connection,name);
   if(name->Port)
      sk->Flags|=SOCKET_CONNECT_PORT;
   if(name->Network)
      sk->Flags|=SOCKET_CONNECT_NETWORK;
   if(memcmp(name->Node,BroadcastNode,IPX_NODE_LEN))
      sk->Flags|=SOCKET_CONNECT_NODE;

   return 0;
}

/* -------------------------------------------------------------------------- */

int ipx_getport(struct Socket *sk)
{  return sk->Port; }

int ipx_get_mtu(register struct RiscIPX *ri)
{  return ri->MTU; }


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Transmit and receive functions.                                      |
   |                                                                        |
   +------------------------------------------------------------------------+ */

struct mbuf *ipx_recv_mbuf(struct Socket *sk,register struct RiscIPX *ri)
{  struct mbuf *mb;

   #if defined DEBUG && (DEBUG&128)
   Trace("\n recvmbuf ");
   #endif

   Disable(ri);

   mb=sk->ReadQueue;
   if(mb)
      if((sk->ReadQueue=mb->list)==NULL)
      {  sk->TailQueue=NULL; }
   Enable(ri);

   #if defined DEBUG && !(DEBUG&128)
   if(mb)
      Trace("\n recvmbuf ");
   #endif
   
   #if defined DEBUG && (DEBUG&64) && (DEBUG&128)
   if(mb)
      DumpPacket(mb->next,"\n  recv_mbuf from ",(char *)mb+mb->off+8,ri->mbctl.count_bytes(&ri->mbctl,mb->next),ri->FrameType);
   #endif

   return mb;
}

/* -------------------------------------------------------------------------- */

int ipx_recv(struct Socket *sk,void *buffer,int length,register struct RiscIPX *ri)
{  struct mbuf *mb;
   int n;

   #ifdef DEBUG
   Trace("\n recv ");
   #endif

   if((mb=ipx_recv_mbuf(sk,ri))==NULL)
      return IPX_NO_PACKETS_READY;

   n=ri->mbctl.count_bytes(&ri->mbctl,mb);
   ri->mbctl.export(&ri->mbctl,mb,length,buffer);
   ri->mbctl.freem(&ri->mbctl,mb);

   return n;
}

/* -------------------------------------------------------------------------- */

int ipx_send_mbuf(struct Socket *sk,struct mbuf *datamb,register struct RiscIPX *ri)
{  struct Route *rt=NULL;
   struct mbuf *hdrmb;
   struct UnpackedHeader header;
   char node[IPX_NODE_LEN];
   int r,ft;
   static char headersize[6]={0,PACKETHEADERSIZE+3,PACKETHEADERSIZE,PACKETHEADERSIZE+8,PACKETHEADERSIZE,PACKETHEADERSIZE };

   #ifdef DEBUG
   Trace("\n sendmbuf(");
   TraceVal((int)sk);
   TraceVal((int)datamb);
   TraceVal((int)ri);
   Trace(") ");
   #endif

   ft=ri->FrameType;

   if(sk->Flags&SOCKET_CONNECT_NETWORK)
   {  
      if((rt=ipxrtr_lookup(sk->Connection.Network,ri))==NULL)
      {  ri->mbctl.freem(&ri->mbctl,datamb);
         #ifdef DEBUG
         Trace("\n no route to net");
         #endif
         return IPX_NO_ROUTE_TO_NETWORK;  }

      if((rt->Flags&ROUTE_FRAMEMASK)!=FRAMETYPE_AUTO)
         ft=rt->Flags&ROUTE_FRAMEMASK;
   }

   if(ft==FRAMETYPE_AUTO)
   {
      if(!(sk->Flags&SOCKET_CONNECT_NETWORK))
      {  
         ft=(ri->Flags&IPX_FT_CYCLE_MASK)+1;
         if(ft>FRAMETYPE_LAST)
            ft=FRAMETYPE_FIRST;
         ri->Flags=(ri->Flags&~IPX_FT_CYCLE_MASK)|ft;
      }
      else
         return IPX_NO_ROUTE_TO_NETWORK;
   }
   
   if((hdrmb=ri->mbctl.alloc(&ri->mbctl,headersize[ft],NULL))==NULL)
   {  ri->mbctl.freem(&ri->mbctl,datamb);
      return IPX_NOT_ENOUGH_MEMORY; }

   header.CheckSum=0xffff;
   header.Length=ri->mbctl.count_bytes(&ri->mbctl,datamb)+PACKETHEADERSIZE;
   header.PacketType=sk->SendPT;

   header.Source.Network=ipxrtr_get_local_net_number(ft,ri);
   #ifdef ECONET
   memcpy_node(header.Source.Node,ft==FRAMETYPE_ECONET?ri->EconetNode:ri->Node);
   #else
   memcpy_node(header.Source.Node,ri->Node);
   #endif

   memcpy_node(header.Destination.Node,
          sk->Flags&SOCKET_CONNECT_NETWORK?sk->Connection.Node:BroadcastNode);    /* sk..Node may be broadcast node */

   if(sk->Flags&SOCKET_CONNECT_NETWORK)
   {  header.Destination.Network=sk->Connection.Network;
      memcpy_node(node,rt->Hops?rt->Node:header.Destination.Node); }
   else
   {  header.Destination.Network=header.Source.Network;
      memcpy_node(node,header.Destination.Node); }

   header.Source.Port=sk->Port;
   header.Destination.Port=(sk->Connection.Port)||(sk->Flags&SOCKET_CONNECT_PORT)?
                           sk->Connection.Port:0xffff;

   hdrmb->next=datamb;
   hdrmb->list=NULL;

   if((r=Pack(hdrmb,&header,Format[ft]))<0)
   {  ri->mbctl.freem(&ri->mbctl,hdrmb);
      return r; }

   #if defined DEBUG && (DEBUG&64)
   DumpPacket(hdrmb,"\n  send to ",node,headersize[ft]+header.Length-PACKETHEADERSIZE,ft);
   #endif

   #ifdef ECONET
   if(ft==FRAMETYPE_ECONET)
   {  char *p;
      
      if((p=malloc(header.Length))==NULL)
      {  ri->mbctl.freem(&ri->mbctl,hdrmb);
         return IPX_NOT_ENOUGH_MEMORY; }
      
      ri->mbctl.export(&ri->mbctl,hdrmb,header.Length,p);
      ri->mbctl.freem(&ri->mbctl,hdrmb);

      Trace("\n Econet_DoTransmit(packet type="); TraceByte(header.PacketType);
      Trace(",node=");     TraceByte(node[5]);
      Trace(",network=");  TraceByte(node[0]?0xfe:header.Source.Network);
      Trace(",address=");  TraceVal((int)(p+2*sizeof(struct Address)-PACKETHEADERSIZE));
      Trace(",length=");   TraceVal(header.Length-ECONET_PREHEADER);
      Trace(")");

      r=Econet_DoTransmit(header.PacketType,ECONET_IPX_PORT,node[5],
                          node[0]?0xfe:header.Source.Network,
                          p+ECONET_PREHEADER,header.Length-ECONET_PREHEADER,
                          2,2);

      free(p);
   }
   else
   #endif
   {
      r=(int)Transmit(ri->Unit,ft==FRAMETYPE_E2?0x8137:
                      headersize[ft]+header.Length-PACKETHEADERSIZE,
                      hdrmb,node);
   }

   return r?IPX_TRANSMIT_ERROR:0;
}

/* -------------------------------------------------------------------------- */

int ipx_send_unsafe(struct Socket *sk,void *data,int length,register struct RiscIPX *ri)
{  struct mbuf *mb;

   if((mb=ri->mbctl.alloc_u(&ri->mbctl,length,data))==NULL)   /* uses unsafe data for speedup */
      return IPX_NOT_ENOUGH_MEMORY;

   return ipx_send_mbuf(sk,mb,ri);
}

/* -------------------------------------------------------------------------- */

int ipx_send(struct Socket *sk,void *data,int length,register struct RiscIPX *ri)
{  struct mbuf *mb;

   if((mb=ri->mbctl.alloc(&ri->mbctl,length,data))==NULL)   /* copies data for normal send */
      return IPX_NOT_ENOUGH_MEMORY;

   return ipx_send_mbuf(sk,mb,ri);
}

/* -------------------------------------------------------------------------- */

int ipx_send_pack(struct Socket *sk,void *data,char *format,register struct RiscIPX *ri)
{  struct mbuf *mb;
   int i;

   if((i=CountPack(format,data))<0)
      return i;

   if((mb=ri->mbctl.alloc(&ri->mbctl,i,NULL))==NULL)
      return IPX_NOT_ENOUGH_MEMORY;

   if((i=Pack(mb,data,format))<0)
   {  ri->mbctl.freem(&ri->mbctl,mb);
      return i; }

   return ipx_send_mbuf(sk,mb,ri);
}


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Star commands.                                                       |
   |                                                                        |
   +------------------------------------------------------------------------+ */

static char FrameNames[]=
{  
   "Auto\0\0"
   "802.2\0"
   "802.3\0"
   "SNAP\0\0"
   "II\0\0\0\0"
   #ifdef ECONET
   "Econet"
   #endif
};

void PutsType(int type)
{
   #ifdef ECONET
   if(type && type!=FRAMETYPE_ECONET)
   #else
   if(type)
   #endif
      riscos_puts("Ethernet_");
   riscos_puts(FrameNames+type*6);
}

struct kernel_error *StarStatus(const char *commandTail,int numParms,struct RiscIPX *ri)
{  struct Socket *sk;
   char buffer[16];

   #ifdef ECONET
   riscos_puts("EcoRiscIPX [");
   #else
   riscos_puts("RiscIPX [");
   #endif
   memcpy(buffer,Version,4);
   buffer[4]=0;
   riscos_puts(buffer);
   riscos_puts("]\r\nFrame type: ");
   PutsType(ri->FrameType);

   riscos_puts("\r\nNetwork SWI: &");
   ConvertHex8(ri->SwiBase,buffer);
   riscos_puts(buffer+2);
   
   riscos_puts("\r\n"
               "Sock Connection                 AP SP\r\n"
               "=====================================\r\n");

   forList(&ri->Sockets,sk)
   {
      if(sk->Flags&SOCKET_BOUND)
         putshort(sk->Port);
      else
         riscos_puts("xxxx");

      riscos_puts(" ");
      if(sk->Flags&SOCKET_CONNECT_NETWORK)
         putlong(sk->Connection.Network);
      else
         riscos_puts("xxxxxxxx");
      riscos_puts(":");

      if(sk->Flags&SOCKET_CONNECT_NODE)
         putaddress(sk->Connection.Node);
      else
         riscos_puts("xxxxxxxxxxxx");
      riscos_puts(":");

      if(sk->Flags&SOCKET_CONNECT_PORT)
         putshort(sk->Connection.Port);
      else
         riscos_puts("xxxx");
      riscos_puts(" ");

      if(sk->Flags&SOCKET_ANY_TYPE)
         riscos_puts("xx");
      else
         putbyte(sk->AcceptPT);
      riscos_puts(" ");
      putbyte(sk->SendPT);

      riscos_puts("\r\n");
   }

   return NULL;
}


/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Utility subroutines.                                                 |
   |                                                                        |
   +------------------------------------------------------------------------+ */

#define tolower(c)  (((c)>='A' && (c)<='Z')?(c)+'a'-'A':(c))

int strcmp(const char *p,const char *q)
{
   while(tolower(*p)==tolower(*q))
   {  if(!*p)
         return 0;
      p++;
      q++; }

   return tolower(*q)-tolower(*p);
}

/* -------------------------------------------------------------------------- */

static int MatchFrame(char *buffer)
{
   if(!strcmp(buffer,FrameNames+6*FRAMETYPE_8022))
      return FRAMETYPE_8022;
   if(!strcmp(buffer,FrameNames+6*FRAMETYPE_8023))
      return FRAMETYPE_8023;
   if(!strcmp(buffer,FrameNames+6*FRAMETYPE_E2) || !strcmp(buffer,"2"))
      return FRAMETYPE_E2;
   if(!strcmp(buffer,FrameNames+6*FRAMETYPE_SNAP))
      return FRAMETYPE_SNAP;
   #ifdef ECONET
   if(!strcmp(buffer,FrameNames+6*FRAMETYPE_ECONET))
      return FRAMETYPE_ECONET;
   #endif
   return FRAMETYPE_AUTO;
}

/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Convert IPX address to and from ASCII.                               |
   |                                                                        |
   +------------------------------------------------------------------------+ */

static int UnHex(int i)
{  i-='0';
   if(i<0)
      return -1;
   if(i>9)
   {  i-=7;
      if(i>15)
      {  i-=32;
         if(i<10)
            return -1;
      }
      if(i>15)
         return -1;
   }
   return i;
}

/* -------------------------------------------------------------------------- */

int AtoIPX(struct Address *address,const char *string)
{  int i,flag;
   long network;
   char node[IPX_NODE_LEN]={0};
   unsigned short port;

   network=flag=0;
   while(*string && *string!=':')
   { 
      i=UnHex(*string++);
      if(i<0)
         return i;
      network=(network<<4)|i;
      flag=1;
   }
   if(flag)
      address->Network=network;
   
   if(!*string)
      return 0;
   if(*string++!=':')
      return -1;

   flag=0;
   while(*string && *string!=':')
   { 
      i=UnHex(*string++);
      if(i<0)
         return i;
      for(flag=0;flag<6;flag++)
      {  node[flag]<<=4;
         if(flag<5)
            node[flag]|=(node[flag+1]>>4)&15;
      }
      node[5]|=i;
   }
   if(flag)
      memcpy_node(address->Node,node);

   if(!*string)
      return 0;
   if(*string++!=':')
      return -1;

   port=flag=0;
   while(*string && *string!=':')
   { 
      i=UnHex(*string++);
      if(i<0)
         return i;
      port=(port<<4)|i;
      flag=1;
   }
   if(flag)
      address->Port=port;
   
   return 0;
}

/* +------------------------------------------------------------------------+
   |                                                                        |
   |   Dump packet to trace output.                                         |
   |                                                                        |
   +------------------------------------------------------------------------+ */

#if defined DEBUG && (DEBUG&64 || DEBUG&128)
static void DumpByte(int b)
{  TraceChar(((b>>4)&15)>9?((b>>4)&15)+'a'-10:((b>>4)&15)+'0');
   TraceChar((b&15)>9?(b&15)+'a'-10:(b&15)+'0'); }

#if 0
static void DumpShort(int b)
{  DumpByte(b>>8);
   DumpByte(b);
}
#endif

static void DumpInt(int i)
{  int c,n,f=0;

   if(i<0 || i>=10000)
      i=9999;

   for(n=1000;n>1;)
   {  for(c='0';i>=n;c++)
         i-=n;
      if(f || c>'0')
      {  f=1;
         TraceChar(c); }
      switch(n)
      {  case 1000:
            n=100;
            break;
         case 100:
            n=10;
            break;
         case 10:
            n=1;
            break;
      }
   }
   TraceChar(i+'0');
}

static void DumpMAC(char *mac)
{  DumpByte(*mac++);
   DumpByte(*mac++);
   DumpByte(*mac++);
   DumpByte(*mac++);
   DumpByte(*mac++);
   DumpByte(*mac);
}

static void DumpIPXaddr(char *addr)
{
   DumpByte(addr[0]);
   DumpByte(addr[1]);
   DumpByte(addr[2]);
   DumpByte(addr[3]);
   TraceChar(':');
   DumpMAC(addr+4);
   TraceChar(':');
   DumpByte(addr[10]);
   DumpByte(addr[11]);
}

static void DumpLine(char *bytes,int offset,int len)
{  int i;

   TraceChar('\n');
   TraceChar((offset>>8)+'0');
   DumpByte(offset);
   Trace(": ");

   for(i=0;i<len;i++)
   {  DumpByte(bytes[i]);
      Trace(i==7?"  ":" "); }
   for(;i<16;i++)
      Trace(i==7?"    ":"   ");
   Trace("  ");
   for(i=0;i<len;i++)
      TraceChar(bytes[i]<32?'.':bytes[i]);
}

static void DumpPacket(struct mbuf *mb,char *title,char *mac,int bytes,int frame)
{  int x,y,l,off=0;
   char *p,line[10+PACKETHEADERSIZE];
   struct mbuf *mp;

   Trace(title);
   DumpMAC(mac);
   Trace("  bytes=");
   DumpInt(bytes);

   switch(frame)
   {  case FRAMETYPE_AUTO:
         break;
      case FRAMETYPE_8022:
         Trace(" 802.2");
         off=3;
         break;
      case FRAMETYPE_SNAP:
         Trace(" SNAP");
         off=8;
         break;
      default:
         Trace(" 802.3");
         off=0;
         break;
   }

   mp=mb;
   x=0;
   while(x<off+PACKETHEADERSIZE)
   {
      p=(char *)mp+mp->off;
      l=mp->len;
      while(l && x<off+PACKETHEADERSIZE)
      {  
         if(x>=off)
            line[x-off]=*p;
         p++;
         x++;
         l--;
      }
      mp=mp->next;
   }

   if(frame!=FRAMETYPE_AUTO)
   {
      TraceChar('\n');
      Trace("IPX:");
      TraceByte(line[5]);
      Trace("  ");
      DumpIPXaddr(line+18);
      Trace(" -> ");
      DumpIPXaddr(line+6);
   }

   x=y=0;
   while(bytes)
   {
      p=(char *)mb+mb->off;
      l=mb->len;
      while(l && bytes)
      {  
         while(x<16 && l && bytes)
         {  line[x]=*p++;
            x++;
            l--;
            bytes--; }
         if(!bytes || x==16)
         {  DumpLine(line,y,x);
            y+=16;
            x=0; }
      }
      if(!l)
         mb=mb->next;
   }
   TraceChar('\n');
}
#endif

/* ----- EOF RISCIPX.C ----- */

